<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\AnalyticsService;
use App\Models\User;
use App\Models\SealAnalytics;
use Carbon\Carbon;
use Exception;

class ProcessAnalytics extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'analytics:process 
                            {--cleanup : Clean up old analytics data}
                            {--report : Generate analytics reports}
                            {--days=90 : Number of days to keep analytics data}
                            {--user= : Process analytics for specific user ID}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process analytics data, generate reports, and clean up old data';

    /**
     * The analytics service.
     *
     * @var AnalyticsService
     */
    protected $analyticsService;

    /**
     * Create a new command instance.
     */
    public function __construct(AnalyticsService $analyticsService)
    {
        parent::__construct();
        $this->analyticsService = $analyticsService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Processing analytics data...');
        
        $cleanup = $this->option('cleanup');
        $generateReport = $this->option('report');
        $keepDays = (int) $this->option('days');
        $userId = $this->option('user');
        
        try {
            if ($cleanup) {
                $this->performCleanup($keepDays);
            }
            
            if ($generateReport) {
                $this->generateReports($userId);
            }
            
            if (!$cleanup && !$generateReport) {
                $this->info('No specific action requested. Use --cleanup or --report options.');
                $this->showAnalyticsSummary();
            }
            
            return 0;
            
        } catch (Exception $e) {
            $this->error("Error processing analytics: {$e->getMessage()}");
            return 1;
        }
    }
    
    /**
     * Clean up old analytics data.
     */
    protected function performCleanup(int $keepDays): void
    {
        $this->info("Cleaning up analytics data older than {$keepDays} days...");
        
        $cutoffDate = Carbon::now()->subDays($keepDays);
        
        $deletedCount = SealAnalytics::where('viewed_at', '<', $cutoffDate)->delete();
        
        $this->info("Deleted {$deletedCount} old analytics records.");
        
        // Clean up any orphaned records or perform other maintenance
        $this->performMaintenance();
    }
    
    /**
     * Generate analytics reports.
     */
    protected function generateReports(?string $userId): void
    {
        $this->info('Generating analytics reports...');
        
        if ($userId) {
            $user = User::find($userId);
            if (!$user) {
                $this->error("User with ID {$userId} not found.");
                return;
            }
            
            $this->generateUserReport($user);
        } else {
            $this->generateSystemReport();
        }
    }
    
    /**
     * Generate report for a specific user.
     */
    protected function generateUserReport(User $user): void
    {
        $this->info("Generating report for user: {$user->name} ({$user->email})");
        
        $analytics = $this->analyticsService->getUserAnalytics($user->id, 'month');
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['Total Views', number_format($analytics['total_views'])],
                ['Unique Visitors', number_format($analytics['unique_visitors'])],
                ['Active Domains', count($user->domains->where('is_active', true))],
                ['Active Seals', count($user->domains->flatMap->trustSeals->where('is_active', true))],
                ['Top Country', $analytics['top_countries'][0]['country'] ?? 'N/A'],
                ['Avg Security Score', number_format($user->getAverageSecurityScore(), 1)]
            ]
        );
        
        // Show top performing domains
        if (!empty($analytics['domain_performance'])) {
            $this->info('\nTop Performing Domains:');
            $domainData = [];
            foreach (array_slice($analytics['domain_performance'], 0, 5) as $domain) {
                $domainData[] = [$domain['domain'], number_format($domain['views'])];
            }
            $this->table(['Domain', 'Views'], $domainData);
        }
    }
    
    /**
     * Generate system-wide report.
     */
    protected function generateSystemReport(): void
    {
        $this->info('Generating system-wide analytics report...');
        
        // Get overall statistics
        $totalUsers = User::count();
        $totalViews = SealAnalytics::count();
        $totalViewsToday = SealAnalytics::whereDate('viewed_at', Carbon::today())->count();
        $totalViewsThisMonth = SealAnalytics::whereMonth('viewed_at', Carbon::now()->month)
            ->whereYear('viewed_at', Carbon::now()->year)
            ->count();
        
        $this->table(
            ['System Metric', 'Value'],
            [
                ['Total Users', number_format($totalUsers)],
                ['Total Views (All Time)', number_format($totalViews)],
                ['Views Today', number_format($totalViewsToday)],
                ['Views This Month', number_format($totalViewsThisMonth)],
                ['Avg Views per User', $totalUsers > 0 ? number_format($totalViews / $totalUsers, 1) : '0']
            ]
        );
        
        // Top countries
        $topCountries = SealAnalytics::selectRaw('country_code, COUNT(*) as views')
            ->whereNotNull('country_code')
            ->groupBy('country_code')
            ->orderByDesc('views')
            ->limit(5)
            ->get();
            
        if ($topCountries->isNotEmpty()) {
            $this->info('\nTop Countries:');
            $countryData = [];
            foreach ($topCountries as $country) {
                $countryData[] = [$country->country_code, number_format($country->views)];
            }
            $this->table(['Country', 'Views'], $countryData);
        }
    }
    
    /**
     * Show analytics summary.
     */
    protected function showAnalyticsSummary(): void
    {
        $this->info('\nAnalytics Summary:');
        
        $today = SealAnalytics::whereDate('viewed_at', Carbon::today())->count();
        $yesterday = SealAnalytics::whereDate('viewed_at', Carbon::yesterday())->count();
        $thisWeek = SealAnalytics::whereBetween('viewed_at', [
            Carbon::now()->startOfWeek(),
            Carbon::now()->endOfWeek()
        ])->count();
        
        $this->table(
            ['Period', 'Views'],
            [
                ['Today', number_format($today)],
                ['Yesterday', number_format($yesterday)],
                ['This Week', number_format($thisWeek)]
            ]
        );
    }
    
    /**
     * Perform general maintenance tasks.
     */
    protected function performMaintenance(): void
    {
        $this->info('Performing maintenance tasks...');
        
        // Update any cached statistics or perform other maintenance
        // This could include updating aggregate tables, clearing caches, etc.
        
        $this->info('Maintenance completed.');
    }
}