<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\SecurityMonitoringService;
use App\Models\Domain;
use Exception;

class RunSecurityChecks extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'security:check-all 
                            {--domain=* : Specific domains to check}
                            {--force : Force check even if recently checked}
                            {--delay=5 : Delay between checks in seconds}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Run security checks for all verified domains';

    /**
     * The security monitoring service.
     *
     * @var SecurityMonitoringService
     */
    protected $securityService;

    /**
     * Create a new command instance.
     */
    public function __construct(SecurityMonitoringService $securityService)
    {
        parent::__construct();
        $this->securityService = $securityService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting security checks...');
        
        $specificDomains = $this->option('domain');
        $forceCheck = $this->option('force');
        $delay = (int) $this->option('delay');
        
        try {
            if (!empty($specificDomains)) {
                // Check specific domains
                $domains = Domain::whereIn('domain', $specificDomains)
                    ->where('status', 'verified')
                    ->get();
                    
                if ($domains->isEmpty()) {
                    $this->error('No verified domains found matching the specified criteria.');
                    return 1;
                }
            } else {
                // Get domains that need security checks
                $domains = $forceCheck 
                    ? Domain::where('status', 'verified')->get()
                    : $this->securityService->getDomainsForSecurityCheck();
            }
            
            $this->info("Found {$domains->count()} domains to check.");
            
            if ($domains->isEmpty()) {
                $this->info('No domains need security checks at this time.');
                return 0;
            }
            
            $progressBar = $this->output->createProgressBar($domains->count());
            $progressBar->start();
            
            $successCount = 0;
            $errorCount = 0;
            $errors = [];
            
            foreach ($domains as $domain) {
                try {
                    $this->securityService->runSecurityCheck($domain);
                    $successCount++;
                    
                    $this->line('');
                    $this->info("✓ Completed security check for: {$domain->domain}");
                    
                } catch (Exception $e) {
                    $errorCount++;
                    $errors[] = [
                        'domain' => $domain->domain,
                        'error' => $e->getMessage()
                    ];
                    
                    $this->line('');
                    $this->error("✗ Failed security check for: {$domain->domain} - {$e->getMessage()}");
                }
                
                $progressBar->advance();
                
                // Add delay between checks to avoid overwhelming external services
                if ($delay > 0 && !$domain->is($domains->last())) {
                    sleep($delay);
                }
            }
            
            $progressBar->finish();
            $this->line('');
            
            // Summary
            $this->info('Security checks completed!');
            $this->table(
                ['Status', 'Count'],
                [
                    ['Successful', $successCount],
                    ['Failed', $errorCount],
                    ['Total', $domains->count()]
                ]
            );
            
            // Show errors if any
            if (!empty($errors)) {
                $this->line('');
                $this->error('Errors encountered:');
                foreach ($errors as $error) {
                    $this->line("• {$error['domain']}: {$error['error']}");
                }
            }
            
            return $errorCount > 0 ? 1 : 0;
            
        } catch (Exception $e) {
            $this->error("Fatal error during security checks: {$e->getMessage()}");
            return 1;
        }
    }
}