<?php

namespace App\Events;

use App\Models\TrustSeal;
use Illuminate\Broadcasting\Channel;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PresenceChannel;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Http\Request;

class SealViewed
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public $trustSeal;
    public $viewData;
    public $viewedAt;

    /**
     * Create a new event instance.
     */
    public function __construct(TrustSeal $trustSeal, array $viewData = [])
    {
        $this->trustSeal = $trustSeal;
        $this->viewData = $viewData;
        $this->viewedAt = now();
    }
    
    /**
     * Create from HTTP request.
     */
    public static function fromRequest(TrustSeal $trustSeal, Request $request): self
    {
        $userAgent = $request->userAgent();
        
        return new self($trustSeal, [
            'ip_address' => $request->ip(),
            'user_agent' => $userAgent,
            'referrer' => $request->header('referer', 'direct'),
            'country' => 'Unknown', // Will be resolved by AnalyticsService
            'browser' => self::getBrowserFromUserAgent($userAgent),
            'os' => self::getOSFromUserAgent($userAgent),
            'device_type' => self::getDeviceTypeFromUserAgent($userAgent),
            'viewport' => [
                'width' => $request->input('vw'),
                'height' => $request->input('vh')
            ],
            'page_url' => $request->input('page_url', $request->header('referer')),
            'utm_source' => $request->input('utm_source'),
            'utm_medium' => $request->input('utm_medium'),
            'utm_campaign' => $request->input('utm_campaign')
        ]);
    }
    
    /**
     * Extract browser from user agent.
     */
    protected static function getBrowserFromUserAgent(?string $userAgent): string
    {
        if (!$userAgent) return 'Unknown';
        
        $browsers = [
            'Chrome' => '/Chrome\/([0-9.]+)/',
            'Firefox' => '/Firefox\/([0-9.]+)/',
            'Safari' => '/Safari\/([0-9.]+)/',
            'Edge' => '/Edge\/([0-9.]+)/',
            'Opera' => '/Opera\/([0-9.]+)/',
            'Internet Explorer' => '/MSIE ([0-9.]+)/'
        ];
        
        foreach ($browsers as $browser => $pattern) {
            if (preg_match($pattern, $userAgent)) {
                return $browser;
            }
        }
        
        return 'Unknown';
    }
    
    /**
     * Extract OS from user agent.
     */
    protected static function getOSFromUserAgent(?string $userAgent): string
    {
        if (!$userAgent) return 'Unknown';
        
        $os = [
            'Windows' => '/Windows NT/',
            'Mac OS' => '/Mac OS X/',
            'Linux' => '/Linux/',
            'Android' => '/Android/',
            'iOS' => '/(iPhone|iPad)/',
        ];
        
        foreach ($os as $system => $pattern) {
            if (preg_match($pattern, $userAgent)) {
                return $system;
            }
        }
        
        return 'Unknown';
    }
    
    /**
     * Extract device type from user agent.
     */
    protected static function getDeviceTypeFromUserAgent(?string $userAgent): string
    {
        if (!$userAgent) return 'desktop';
        
        if (preg_match('/(tablet|ipad|playbook|silk)|(android(?!.*mobile))/i', $userAgent)) {
            return 'tablet';
        }
        
        if (preg_match('/(mobile|iphone|ipod|blackberry|android|palm|windows\sce)/i', $userAgent)) {
            return 'mobile';
        }
        
        return 'desktop';
    }
    
    /**
     * Get referrer domain.
     */
    public function getReferrerDomain(): ?string
    {
        $referrer = $this->viewData['referrer'] ?? null;
        
        if (!$referrer || $referrer === 'direct') {
            return null;
        }
        
        $parsed = parse_url($referrer);
        return $parsed['host'] ?? null;
    }
    
    /**
     * Check if this is a unique view.
     */
    public function isUniqueView(): bool
    {
        $ip = $this->viewData['ip_address'] ?? null;
        
        if (!$ip) {
            return false;
        }
        
        // Consider it unique if no view from same IP in last 24 hours
        return !\App\Models\SealAnalytics::where('trust_seal_id', $this->trustSeal->id)
            ->where('ip_address', $ip)
            ->where('viewed_at', '>=', now()->subDay())
            ->exists();
    }
    
    /**
     * Get location data for the view.
     */
    public function getLocationData(): array
    {
        // This would integrate with a GeoIP service
        return [
            'country' => $this->viewData['country'] ?? 'Unknown',
            'region' => $this->viewData['region'] ?? null,
            'city' => $this->viewData['city'] ?? null,
            'latitude' => $this->viewData['latitude'] ?? null,
            'longitude' => $this->viewData['longitude'] ?? null
        ];
    }
    
    /**
     * Check if view should be tracked (e.g., not from bots).
     */
    public function shouldTrack(): bool
    {
        $userAgent = $this->viewData['user_agent'] ?? '';
        
        // Common bot patterns
        $botPatterns = [
            '/bot/i',
            '/crawler/i',
            '/spider/i',
            '/scraper/i',
            '/facebook/i',
            '/twitter/i',
            '/linkedin/i',
            '/whatsapp/i',
            '/telegram/i'
        ];
        
        foreach ($botPatterns as $pattern) {
            if (preg_match($pattern, $userAgent)) {
                return false;
            }
        }
        
        return true;
    }
}