<?php

namespace App\Events;

use App\Models\Domain;
use App\Models\SecurityCheck;
use Illuminate\Broadcasting\Channel;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Broadcasting\PresenceChannel;
use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class SecurityAlert implements ShouldBroadcast
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public $domain;
    public $alertType;
    public $alertLevel;
    public $message;
    public $details;
    public $alertedAt;

    /**
     * Create a new event instance.
     */
    public function __construct(Domain $domain, string $alertType, string $alertLevel, string $message, array $details = [])
    {
        $this->domain = $domain;
        $this->alertType = $alertType;
        $this->alertLevel = $alertLevel;
        $this->message = $message;
        $this->details = $details;
        $this->alertedAt = now();
    }
    
    /**
     * Create from security check.
     */
    public static function fromSecurityCheck(SecurityCheck $check): self
    {
        $alertLevel = match($check->status) {
            'passed' => 'info',
            'warning' => 'warning',
            'failed' => 'error',
            default => 'info'
        };
        
        $message = match($check->check_type) {
            'ssl' => $check->status === 'failed' 
                ? "SSL certificate issue detected for {$check->domain->domain}" 
                : "SSL certificate check completed for {$check->domain->domain}",
            'dns' => $check->status === 'failed' 
                ? "DNS configuration issue detected for {$check->domain->domain}" 
                : "DNS check completed for {$check->domain->domain}",
            'malware' => $check->status === 'failed' 
                ? "Potential malware detected on {$check->domain->domain}" 
                : "Malware scan completed for {$check->domain->domain}",
            'safe_browsing' => $check->status === 'failed' 
                ? "Safe browsing warning for {$check->domain->domain}" 
                : "Safe browsing check completed for {$check->domain->domain}",
            default => "Security check completed for {$check->domain->domain}"
        };
        
        return new self(
            $check->domain,
            $check->check_type,
            $alertLevel,
            $message,
            array_merge($check->details ?? [], [
                'check_id' => $check->id,
                'score' => $check->score,
                'checked_at' => $check->checked_at->toISOString()
            ])
        );
    }

    /**
     * Get the channels the event should broadcast on.
     *
     * @return array<int, \Illuminate\Broadcasting\Channel>
     */
    public function broadcastOn(): array
    {
        return [
            new PrivateChannel('user.' . $this->domain->user_id),
            new PrivateChannel('domain.' . $this->domain->id)
        ];
    }
    
    /**
     * The event's broadcast name.
     */
    public function broadcastAs(): string
    {
        return 'security.alert';
    }
    
    /**
     * Get the data to broadcast.
     */
    public function broadcastWith(): array
    {
        return [
            'domain_id' => $this->domain->id,
            'domain' => $this->domain->domain,
            'alert_type' => $this->alertType,
            'alert_level' => $this->alertLevel,
            'message' => $this->message,
            'details' => $this->details,
            'alerted_at' => $this->alertedAt->toISOString(),
            'requires_action' => in_array($this->alertLevel, ['warning', 'error'])
        ];
    }
    
    /**
     * Get the alert priority for notifications.
     */
    public function getPriority(): string
    {
        return match($this->alertLevel) {
            'error' => 'high',
            'warning' => 'medium',
            'info' => 'low',
            default => 'low'
        };
    }
    
    /**
     * Get recommended actions based on alert type.
     */
    public function getRecommendedActions(): array
    {
        return match($this->alertType) {
            'ssl' => [
                'Check SSL certificate expiry date',
                'Verify SSL certificate configuration',
                'Renew SSL certificate if expired',
                'Contact hosting provider if issues persist'
            ],
            'dns' => [
                'Verify DNS records configuration',
                'Check nameserver settings',
                'Contact domain registrar if needed',
                'Update DNS records if required'
            ],
            'malware' => [
                'Scan website files for malware',
                'Update website software and plugins',
                'Change all passwords',
                'Contact security expert if needed'
            ],
            'safe_browsing' => [
                'Check Google Search Console',
                'Scan website for malicious content',
                'Request review after cleaning',
                'Monitor website regularly'
            ],
            default => [
                'Review security check details',
                'Take appropriate action based on findings',
                'Contact support if needed'
            ]
        };
    }
}