<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use App\Models\Domain;
use App\Models\DomainVerification;
use App\Models\TrustSeal;
use App\Services\DomainVerificationService;
use App\Services\SecurityCheckService;

class DomainController extends Controller
{
    protected $verificationService;
    protected $securityService;
    
    public function __construct(DomainVerificationService $verificationService, SecurityCheckService $securityService)
    {
        $this->middleware('auth');
        $this->middleware('verified');
        $this->verificationService = $verificationService;
        $this->securityService = $securityService;
    }

    public function index()
    {
        $user = Auth::user();
        $domains = $user->domains()
            ->with(['trustSeals', 'securityChecks' => function($query) {
                $query->latest()->limit(1);
            }])
            ->latest()
            ->paginate(10);
            
        return view('domains.index', compact('domains'));
    }

    public function create()
    {
        $user = Auth::user();
        
        if (!$user->canAddDomain()) {
            return redirect()->route('domains.index')
                ->with('error', 'You have reached your domain limit. Please upgrade your plan to add more domains.');
        }
        
        return view('domains.create');
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->canAddDomain()) {
            return redirect()->route('domains.index')
                ->with('error', 'You have reached your domain limit.');
        }
        
        $validator = Validator::make($request->all(), [
            'domain' => 'required|string|max:255|unique:domains,domain',
        ]);
        
        // Custom domain validation
        $validator->after(function ($validator) use ($request) {
            $domain = $request->input('domain');
            
            // Remove protocol if present
            $domain = preg_replace('/^https?:\/\//', '', $domain);
            
            // Remove trailing slash
            $domain = rtrim($domain, '/');
            
            // Basic domain validation
            if (!filter_var('http://' . $domain, FILTER_VALIDATE_URL)) {
                $validator->errors()->add('domain', 'Please enter a valid domain name.');
            }
            
            // Check if domain is already registered by another user
            if (Domain::where('domain', $domain)->exists()) {
                $validator->errors()->add('domain', 'This domain has already been registered.');
            }
        });
        
        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        $cleanDomain = preg_replace('/^https?:\/\//', '', $request->domain);
        $cleanDomain = rtrim($cleanDomain, '/');
        
        $domain = Domain::create([
            'user_id' => $user->id,
            'domain' => $cleanDomain,
            'status' => 'pending',
            'verification_token' => Str::random(32),
            'is_active' => true
        ]);
        
        // Run initial SSL check
        $this->securityService->checkSSL($domain);
        
        return redirect()->route('domains.show', $domain)
            ->with('success', 'Domain added successfully! Please proceed with verification.');
    }

    public function show(Domain $domain)
    {
        $this->authorize('view', $domain);
        
        $domain->load([
            'domainVerifications' => function($query) {
                $query->latest();
            },
            'trustSeals',
            'securityChecks' => function($query) {
                $query->latest()->limit(10);
            }
        ]);
        
        $verificationInstructions = [
            'dns' => $this->getVerificationInstructions($domain, 'dns'),
            'file' => $this->getVerificationInstructions($domain, 'file'),
            'meta' => $this->getVerificationInstructions($domain, 'meta')
        ];
        
        return view('domains.show', compact('domain', 'verificationInstructions'));
    }

    public function verify(Request $request, Domain $domain)
    {
        $this->authorize('update', $domain);
        
        $validator = Validator::make($request->all(), [
            'method' => 'required|in:dns,file,meta'
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator);
        }
        
        $method = $request->input('method');
        
        try {
            $result = $this->verificationService->verifyDomain($domain, $method);
            
            if ($result['success']) {
                // Create trust seal if verification successful
                if (!$domain->trustSeals()->exists()) {
                    TrustSeal::create([
                        'domain_id' => $domain->id,
                        'seal_type' => 'trust',
                        'seal_code' => Str::random(16),
                        'is_active' => true
                    ]);
                }
                
                // Run security checks
                $this->securityService->runAllChecks($domain);
                
                return redirect()->back()
                    ->with('success', 'Domain verified successfully! Your trust seal is now active.');
            } else {
                return redirect()->back()
                    ->with('error', $result['message'] ?? 'Verification failed. Please check your setup and try again.');
            }
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'An error occurred during verification: ' . $e->getMessage());
        }
    }

    public function regenerateToken(Domain $domain)
    {
        $this->authorize('update', $domain);
        
        $domain->update([
            'verification_token' => Str::random(32)
        ]);
        
        return redirect()->back()
            ->with('success', 'Verification token regenerated successfully.');
    }

    public function toggleStatus(Domain $domain)
    {
        $this->authorize('update', $domain);
        
        $domain->update([
            'is_active' => !$domain->is_active
        ]);
        
        $status = $domain->is_active ? 'activated' : 'deactivated';
        
        return redirect()->back()
            ->with('success', "Domain has been {$status} successfully.");
    }

    public function destroy(Domain $domain)
    {
        $this->authorize('delete', $domain);
        
        $domain->delete();
        
        return redirect()->route('domains.index')
            ->with('success', 'Domain deleted successfully.');
    }

    public function analytics(Domain $domain)
    {
        $this->authorize('view', $domain);
        
        $domain->load('trustSeals.sealAnalytics');
        
        // Get analytics data
        $analytics = $this->getAnalyticsData($domain);
        
        return view('domains.analytics', compact('domain', 'analytics'));
    }
    
    private function getVerificationInstructions(Domain $domain, string $method): array
    {
        $token = $domain->verification_token;
        $domainName = $domain->domain;
        
        switch ($method) {
            case 'dns':
                return [
                    'type' => 'TXT Record',
                    'name' => '_trustseal',
                    'value' => $token,
                    'instructions' => [
                        'Login to your domain registrar or DNS provider',
                        'Navigate to DNS management section',
                        'Add a new TXT record with the following details:',
                        "Name: _trustseal",
                        "Value: {$token}",
                        'Save the record and wait for DNS propagation (up to 24 hours)'
                    ]
                ];
                
            case 'file':
                return [
                    'filename' => 'trustseal-verification.txt',
                    'content' => $token,
                    'url' => "http://{$domainName}/trustseal-verification.txt",
                    'instructions' => [
                        'Create a file named "trustseal-verification.txt"',
                        "Add the following content to the file: {$token}",
                        "Upload the file to your website's root directory",
                        "Ensure the file is accessible at: http://{$domainName}/trustseal-verification.txt"
                    ]
                ];
                
            case 'meta':
                return [
                    'tag' => "<meta name=\"trustseal-verification\" content=\"{$token}\" />",
                    'instructions' => [
                        'Copy the meta tag provided above',
                        'Add it to the <head> section of your website\'s homepage',
                        'Save and publish your changes',
                        'The tag must be visible on your homepage HTML source'
                    ]
                ];
                
            default:
                return [];
        }
    }
    
    private function getAnalyticsData(Domain $domain)
    {
        $sealIds = $domain->trustSeals->pluck('id')->toArray();
        
        if (empty($sealIds)) {
            return [
                'total_views' => 0,
                'daily_views' => [],
                'top_countries' => [],
                'device_stats' => [],
                'browser_stats' => [],
                'referrer_stats' => []
            ];
        }
        
        // Analytics implementation here
        return [
            'total_views' => $domain->trustSeals->sum('display_count'),
            'daily_views' => [], // Implement based on requirements
            'top_countries' => [],
            'device_stats' => [],
            'browser_stats' => [],
            'referrer_stats' => []
        ];
    }
}