<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;
use Illuminate\Support\Facades\Auth;

class ProfileController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Show the user's profile.
     */
    public function show()
    {
        $user = Auth::user();
        
        return view('profile.show', compact('user'));
    }

    /**
     * Update the user's profile information.
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone' => ['nullable', 'string', 'max:20'],
            'company' => ['nullable', 'string', 'max:255'],
            'website' => ['nullable', 'url', 'max:255'],
            'current_password' => ['nullable', 'current_password'],
            'password' => ['nullable', 'confirmed', Password::defaults()],
            'preferences' => ['nullable', 'array'],
            'preferences.email_notifications' => ['boolean'],
            'preferences.security_alerts' => ['boolean'],
            'preferences.weekly_reports' => ['boolean'],
            'preferences.marketing_emails' => ['boolean']
        ]);

        // Update basic profile information
        $user->fill([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
            'company' => $validated['company'] ?? null,
            'website' => $validated['website'] ?? null,
        ]);

        // Update password if provided
        if (!empty($validated['password'])) {
            $user->password = Hash::make($validated['password']);
        }

        // Update preferences
        if (isset($validated['preferences'])) {
            $currentPreferences = $user->preferences ?? $user->getDefaultPreferences();
            $user->preferences = array_merge($currentPreferences, $validated['preferences']);
        }

        $user->save();

        return redirect()->route('profile.show')
            ->with('success', 'Profile updated successfully!');
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request)
    {
        $request->validate([
            'password' => ['required', 'current_password'],
        ]);

        $user = Auth::user();

        // Cancel active subscriptions
        if ($user->hasActiveSubscription()) {
            $user->currentSubscription->update([
                'status' => 'cancelled',
                'cancelled_at' => now()
            ]);
        }

        // Deactivate all domains and trust seals
        $user->domains()->update(['is_active' => false]);
        $user->domains()->with('trustSeal')->get()->each(function ($domain) {
            if ($domain->trustSeal) {
                $domain->trustSeal->update(['is_active' => false]);
            }
        });

        Auth::logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('home')
            ->with('success', 'Your account has been deleted successfully.');
    }

    /**
     * Upload and update profile avatar.
     */
    public function uploadAvatar(Request $request)
    {
        $request->validate([
            'avatar' => ['required', 'image', 'mimes:jpeg,png,jpg,gif', 'max:2048']
        ]);

        $user = Auth::user();

        // Delete old avatar if exists
        if ($user->avatar && Storage::exists('public/avatars/' . $user->avatar)) {
            Storage::delete('public/avatars/' . $user->avatar);
        }

        // Store new avatar
        $avatar = $request->file('avatar');
        $filename = $user->id . '_' . time() . '.' . $avatar->getClientOriginalExtension();
        $avatar->storeAs('public/avatars', $filename);

        $user->update(['avatar' => $filename]);

        return response()->json([
            'success' => true,
            'message' => 'Avatar updated successfully!',
            'avatar_url' => Storage::url('avatars/' . $filename)
        ]);
    }

    /**
     * Remove profile avatar.
     */
    public function removeAvatar()
    {
        $user = Auth::user();

        if ($user->avatar && Storage::exists('public/avatars/' . $user->avatar)) {
            Storage::delete('public/avatars/' . $user->avatar);
        }

        $user->update(['avatar' => null]);

        return response()->json([
            'success' => true,
            'message' => 'Avatar removed successfully!'
        ]);
    }

    /**
     * Get user's API tokens.
     */
    public function apiTokens()
    {
        $user = Auth::user();
        $tokens = $user->tokens()->orderBy('created_at', 'desc')->get();

        return view('profile.api-tokens', compact('tokens'));
    }

    /**
     * Create a new API token.
     */
    public function createApiToken(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'abilities' => ['array']
        ]);

        $user = Auth::user();
        
        // Check if user can create API tokens (premium feature)
        if (!$user->isPremium()) {
            return redirect()->back()
                ->with('error', 'API tokens are available for premium users only.');
        }

        $abilities = $request->input('abilities', ['*']);
        $token = $user->createToken($request->name, $abilities);

        return redirect()->route('profile.api-tokens')
            ->with('success', 'API token created successfully!')
            ->with('token', $token->plainTextToken);
    }

    /**
     * Revoke an API token.
     */
    public function revokeApiToken(Request $request, $tokenId)
    {
        $user = Auth::user();
        $token = $user->tokens()->where('id', $tokenId)->first();

        if (!$token) {
            return redirect()->back()
                ->with('error', 'Token not found.');
        }

        $token->delete();

        return redirect()->back()
            ->with('success', 'API token revoked successfully!');
    }

    /**
     * Export user data (GDPR compliance).
     */
    public function exportData()
    {
        $user = Auth::user();
        
        $data = [
            'user' => $user->toArray(),
            'domains' => $user->domains()->with(['verifications', 'trustSeal', 'securityChecks'])->get()->toArray(),
            'subscriptions' => $user->subscriptions()->get()->toArray(),
            'seal_analytics' => $user->domains()->with('trustSeal.analytics')->get()
                ->flatMap(function ($domain) {
                    return $domain->trustSeal ? $domain->trustSeal->analytics : [];
                })->toArray()
        ];

        $filename = 'user_data_' . $user->id . '_' . now()->format('Y-m-d_H-i-s') . '.json';
        
        return response()->json($data)
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }
}