<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use App\Models\Subscription;
use App\Models\Domain;
use Carbon\Carbon;

class WebhookController extends Controller
{
    /**
     * Handle Stripe webhooks.
     */
    public function stripe(Request $request): JsonResponse
    {
        $payload = $request->getContent();
        $signature = $request->header('Stripe-Signature');
        $endpointSecret = config('services.stripe.webhook_secret');

        // Verify webhook signature (in production)
        if ($endpointSecret && !$this->verifyStripeSignature($payload, $signature, $endpointSecret)) {
            Log::warning('Invalid Stripe webhook signature');
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        $event = json_decode($payload, true);

        if (!$event || !isset($event['type'])) {
            Log::warning('Invalid Stripe webhook payload');
            return response()->json(['error' => 'Invalid payload'], 400);
        }

        Log::info('Stripe webhook received', [
            'type' => $event['type'],
            'id' => $event['id'] ?? null
        ]);

        try {
            switch ($event['type']) {
                case 'payment_intent.succeeded':
                    $this->handlePaymentSucceeded($event['data']['object']);
                    break;

                case 'payment_intent.payment_failed':
                    $this->handlePaymentFailed($event['data']['object']);
                    break;

                case 'customer.subscription.created':
                    $this->handleSubscriptionCreated($event['data']['object']);
                    break;

                case 'customer.subscription.updated':
                    $this->handleSubscriptionUpdated($event['data']['object']);
                    break;

                case 'customer.subscription.deleted':
                    $this->handleSubscriptionCancelled($event['data']['object']);
                    break;

                case 'invoice.payment_succeeded':
                    $this->handleInvoicePaymentSucceeded($event['data']['object']);
                    break;

                case 'invoice.payment_failed':
                    $this->handleInvoicePaymentFailed($event['data']['object']);
                    break;

                default:
                    Log::info('Unhandled Stripe webhook type', ['type' => $event['type']]);
                    break;
            }

            return response()->json(['status' => 'success']);
        } catch (\Exception $e) {
            Log::error('Error processing Stripe webhook', [
                'type' => $event['type'],
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json(['error' => 'Webhook processing failed'], 500);
        }
    }

    /**
     * Handle successful payment.
     */
    protected function handlePaymentSucceeded(array $paymentIntent): void
    {
        $paymentReference = $paymentIntent['id'];
        $amount = $paymentIntent['amount'] / 100; // Convert from cents
        $currency = strtoupper($paymentIntent['currency']);

        // Find pending subscription by payment reference
        $subscription = Subscription::where('payment_reference', $paymentReference)
            ->where('status', 'pending')
            ->first();

        if ($subscription) {
            $subscription->update([
                'status' => 'active',
                'started_at' => now(),
                'expires_at' => $this->calculateExpiryDate($subscription->plan, $subscription->billing_cycle)
            ]);

            // Update user premium status
            $user = $subscription->user;
            $user->update([
                'is_premium' => true,
                'premium_expires_at' => $subscription->expires_at
            ]);

            Log::info('Subscription activated via payment', [
                'subscription_id' => $subscription->id,
                'user_id' => $user->id,
                'amount' => $amount,
                'currency' => $currency
            ]);
        } else {
            Log::warning('No pending subscription found for payment', [
                'payment_reference' => $paymentReference
            ]);
        }
    }

    /**
     * Handle failed payment.
     */
    protected function handlePaymentFailed(array $paymentIntent): void
    {
        $paymentReference = $paymentIntent['id'];
        $errorMessage = $paymentIntent['last_payment_error']['message'] ?? 'Payment failed';

        // Find subscription by payment reference
        $subscription = Subscription::where('payment_reference', $paymentReference)->first();

        if ($subscription) {
            $subscription->update([
                'status' => 'failed'
            ]);

            Log::info('Subscription payment failed', [
                'subscription_id' => $subscription->id,
                'user_id' => $subscription->user_id,
                'error' => $errorMessage
            ]);

            // TODO: Send notification to user about payment failure
        }
    }

    /**
     * Handle subscription created.
     */
    protected function handleSubscriptionCreated(array $stripeSubscription): void
    {
        $customerId = $stripeSubscription['customer'];
        $subscriptionId = $stripeSubscription['id'];
        $status = $stripeSubscription['status'];
        $currentPeriodEnd = Carbon::createFromTimestamp($stripeSubscription['current_period_end']);

        // Find user by Stripe customer ID (you'd need to store this)
        // For now, we'll log the event
        Log::info('Stripe subscription created', [
            'stripe_subscription_id' => $subscriptionId,
            'customer_id' => $customerId,
            'status' => $status,
            'current_period_end' => $currentPeriodEnd
        ]);
    }

    /**
     * Handle subscription updated.
     */
    protected function handleSubscriptionUpdated(array $stripeSubscription): void
    {
        $subscriptionId = $stripeSubscription['id'];
        $status = $stripeSubscription['status'];
        $currentPeriodEnd = Carbon::createFromTimestamp($stripeSubscription['current_period_end']);

        Log::info('Stripe subscription updated', [
            'stripe_subscription_id' => $subscriptionId,
            'status' => $status,
            'current_period_end' => $currentPeriodEnd
        ]);
    }

    /**
     * Handle subscription cancelled.
     */
    protected function handleSubscriptionCancelled(array $stripeSubscription): void
    {
        $subscriptionId = $stripeSubscription['id'];
        $canceledAt = Carbon::createFromTimestamp($stripeSubscription['canceled_at']);

        // You would find the subscription by Stripe ID and update it
        Log::info('Stripe subscription cancelled', [
            'stripe_subscription_id' => $subscriptionId,
            'canceled_at' => $canceledAt
        ]);
    }

    /**
     * Handle successful invoice payment.
     */
    protected function handleInvoicePaymentSucceeded(array $invoice): void
    {
        $subscriptionId = $invoice['subscription'];
        $amount = $invoice['amount_paid'] / 100;
        $currency = strtoupper($invoice['currency']);

        Log::info('Invoice payment succeeded', [
            'stripe_subscription_id' => $subscriptionId,
            'amount' => $amount,
            'currency' => $currency
        ]);
    }

    /**
     * Handle failed invoice payment.
     */
    protected function handleInvoicePaymentFailed(array $invoice): void
    {
        $subscriptionId = $invoice['subscription'];
        $attemptCount = $invoice['attempt_count'];

        Log::warning('Invoice payment failed', [
            'stripe_subscription_id' => $subscriptionId,
            'attempt_count' => $attemptCount
        ]);
    }

    /**
     * Verify Stripe webhook signature.
     */
    protected function verifyStripeSignature(string $payload, ?string $signature, string $secret): bool
    {
        if (!$signature) {
            return false;
        }

        $elements = explode(',', $signature);
        $signatureHash = null;
        $timestamp = null;

        foreach ($elements as $element) {
            [$key, $value] = explode('=', $element, 2);
            if ($key === 'v1') {
                $signatureHash = $value;
            } elseif ($key === 't') {
                $timestamp = $value;
            }
        }

        if (!$signatureHash || !$timestamp) {
            return false;
        }

        // Check timestamp (should be within 5 minutes)
        if (abs(time() - $timestamp) > 300) {
            return false;
        }

        $expectedSignature = hash_hmac('sha256', $timestamp . '.' . $payload, $secret);
        
        return hash_equals($expectedSignature, $signatureHash);
    }

    /**
     * Calculate subscription expiry date.
     */
    protected function calculateExpiryDate(string $plan, string $billingCycle): Carbon
    {
        switch ($billingCycle) {
            case 'monthly':
                return now()->addMonth();
            case 'quarterly':
                return now()->addMonths(3);
            case 'yearly':
                return now()->addYear();
            default:
                return now()->addMonth();
        }
    }

    /**
     * Handle generic webhooks (for other services).
     */
    public function generic(Request $request, string $service): JsonResponse
    {
        $payload = $request->all();
        
        Log::info('Generic webhook received', [
            'service' => $service,
            'payload' => $payload
        ]);

        try {
            switch ($service) {
                case 'security-monitor':
                    $this->handleSecurityWebhook($payload);
                    break;
                    
                case 'analytics':
                    $this->handleAnalyticsWebhook($payload);
                    break;
                    
                default:
                    Log::info('Unhandled webhook service', ['service' => $service]);
                    break;
            }

            return response()->json(['status' => 'success']);
        } catch (\Exception $e) {
            Log::error('Error processing generic webhook', [
                'service' => $service,
                'error' => $e->getMessage()
            ]);

            return response()->json(['error' => 'Webhook processing failed'], 500);
        }
    }

    /**
     * Handle security monitoring webhooks.
     */
    protected function handleSecurityWebhook(array $payload): void
    {
        if (isset($payload['domain']) && isset($payload['security_alerts'])) {
            $domain = Domain::where('domain', $payload['domain'])->first();
            
            if ($domain) {
                // Update security information
                $domain->update([
                    'security_score' => $payload['security_score'] ?? $domain->security_score,
                    'last_security_check' => now()
                ]);

                Log::info('Security webhook processed', [
                    'domain' => $payload['domain'],
                    'score' => $payload['security_score'] ?? 'N/A'
                ]);
            }
        }
    }

    /**
     * Handle analytics webhooks.
     */
    protected function handleAnalyticsWebhook(array $payload): void
    {
        // Handle external analytics data
        Log::info('Analytics webhook processed', $payload);
    }
}