<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class SubscriptionRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return auth()->check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        $availablePlans = ['basic', 'professional', 'enterprise'];
        $billingCycles = ['monthly', 'yearly'];
        
        return [
            'plan' => [
                'required',
                'string',
                Rule::in($availablePlans)
            ],
            'billing_cycle' => [
                'required',
                'string',
                Rule::in($billingCycles)
            ],
            'payment_method' => [
                'required',
                'string',
                Rule::in(['stripe', 'paypal', 'bank_transfer'])
            ],
            'coupon_code' => [
                'nullable',
                'string',
                'max:50',
                'alpha_num'
            ],
            'auto_renew' => [
                'nullable',
                'boolean'
            ],
            'upgrade_immediately' => [
                'nullable',
                'boolean'
            ]
        ];
    }
    
    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'plan.required' => 'Please select a subscription plan.',
            'plan.in' => 'Please select a valid subscription plan.',
            'billing_cycle.required' => 'Please select a billing cycle.',
            'billing_cycle.in' => 'Please select a valid billing cycle.',
            'payment_method.required' => 'Please select a payment method.',
            'payment_method.in' => 'Please select a valid payment method.',
            'coupon_code.alpha_num' => 'Coupon code can only contain letters and numbers.'
        ];
    }
    
    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Normalize plan name
        if ($this->plan) {
            $this->merge([
                'plan' => strtolower(trim($this->plan))
            ]);
        }
        
        // Normalize billing cycle
        if ($this->billing_cycle) {
            $this->merge([
                'billing_cycle' => strtolower(trim($this->billing_cycle))
            ]);
        }
        
        // Normalize coupon code
        if ($this->coupon_code) {
            $this->merge([
                'coupon_code' => strtoupper(trim($this->coupon_code))
            ]);
        }
        
        // Set defaults
        $this->merge([
            'auto_renew' => $this->boolean('auto_renew', true),
            'upgrade_immediately' => $this->boolean('upgrade_immediately', false)
        ]);
    }
    
    /**
     * Get the plan pricing information.
     */
    public function getPlanPricing(): array
    {
        $pricing = [
            'basic' => [
                'monthly' => 9.99,
                'yearly' => 99.99
            ],
            'professional' => [
                'monthly' => 29.99,
                'yearly' => 299.99
            ],
            'enterprise' => [
                'monthly' => 99.99,
                'yearly' => 999.99
            ]
        ];
        
        $plan = $this->input('plan');
        $cycle = $this->input('billing_cycle');
        
        return [
            'plan' => $plan,
            'billing_cycle' => $cycle,
            'amount' => $pricing[$plan][$cycle] ?? 0,
            'currency' => 'USD',
            'discount_percentage' => $cycle === 'yearly' ? 17 : 0 // Yearly plans save ~17%
        ];
    }
    
    /**
     * Check if the user can subscribe to this plan.
     */
    public function canSubscribe(): array
    {
        $user = auth()->user();
        $requestedPlan = $this->input('plan');
        
        // Check if user has an active subscription
        if ($user->hasActiveSubscription()) {
            $currentPlan = $user->currentSubscription->plan;
            
            // Define plan hierarchy
            $planHierarchy = ['basic' => 1, 'professional' => 2, 'enterprise' => 3];
            
            if ($planHierarchy[$requestedPlan] <= $planHierarchy[$currentPlan]) {
                return [
                    'can_subscribe' => false,
                    'reason' => 'Cannot downgrade or subscribe to the same plan'
                ];
            }
        }
        
        return ['can_subscribe' => true];
    }
    
    /**
     * Apply coupon discount if valid.
     */
    public function applyCoupon(float $baseAmount): array
    {
        $couponCode = $this->input('coupon_code');
        
        if (!$couponCode) {
            return [
                'original_amount' => $baseAmount,
                'discount_amount' => 0,
                'final_amount' => $baseAmount,
                'coupon_applied' => false
            ];
        }
        
        // Simple coupon system (in production, this would be in a database)
        $coupons = [
            'WELCOME10' => ['type' => 'percentage', 'value' => 10, 'active' => true],
            'SAVE20' => ['type' => 'percentage', 'value' => 20, 'active' => true],
            'FIRST50' => ['type' => 'fixed', 'value' => 50, 'active' => true]
        ];
        
        if (!isset($coupons[$couponCode]) || !$coupons[$couponCode]['active']) {
            return [
                'original_amount' => $baseAmount,
                'discount_amount' => 0,
                'final_amount' => $baseAmount,
                'coupon_applied' => false,
                'error' => 'Invalid or expired coupon code'
            ];
        }
        
        $coupon = $coupons[$couponCode];
        $discountAmount = 0;
        
        if ($coupon['type'] === 'percentage') {
            $discountAmount = $baseAmount * ($coupon['value'] / 100);
        } else {
            $discountAmount = min($coupon['value'], $baseAmount);
        }
        
        return [
            'original_amount' => $baseAmount,
            'discount_amount' => $discountAmount,
            'final_amount' => max(0, $baseAmount - $discountAmount),
            'coupon_applied' => true,
            'coupon_code' => $couponCode
        ];
    }
}