<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class UpdateProfileRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return auth()->check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        $user = auth()->user();
        
        return [
            'name' => [
                'required',
                'string',
                'max:255',
                'regex:/^[a-zA-Z\s\-\']+$/'
            ],
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->ignore($user->id)
            ],
            'phone' => [
                'nullable',
                'string',
                'max:20',
                'regex:/^[\+]?[1-9][\d\s\-\(\)]{7,18}$/'
            ],
            'company' => [
                'nullable',
                'string',
                'max:255'
            ],
            'website' => [
                'nullable',
                'string',
                'max:255',
                'url'
            ],
            'current_password' => [
                'nullable',
                'required_with:password',
                'current_password'
            ],
            'password' => [
                'nullable',
                'confirmed',
                Password::min(8)
                    ->letters()
                    ->mixedCase()
                    ->numbers()
                    ->symbols()
            ],
            'password_confirmation' => [
                'nullable',
                'required_with:password'
            ],
            'preferences' => [
                'nullable',
                'array'
            ],
            'preferences.notifications' => [
                'nullable',
                'array'
            ],
            'preferences.notifications.email_alerts' => [
                'nullable',
                'boolean'
            ],
            'preferences.notifications.security_alerts' => [
                'nullable',
                'boolean'
            ],
            'preferences.notifications.weekly_reports' => [
                'nullable',
                'boolean'
            ],
            'preferences.notifications.marketing' => [
                'nullable',
                'boolean'
            ],
            'preferences.dashboard' => [
                'nullable',
                'array'
            ],
            'preferences.dashboard.show_tutorials' => [
                'nullable',
                'boolean'
            ],
            'preferences.dashboard.compact_view' => [
                'nullable',
                'boolean'
            ],
            'preferences.privacy' => [
                'nullable',
                'array'
            ],
            'preferences.privacy.public_profile' => [
                'nullable',
                'boolean'
            ],
            'preferences.privacy.analytics_sharing' => [
                'nullable',
                'boolean'
            ]
        ];
    }
    
    /**
     * Get custom messages for validator errors.
     */
    public function messages(): array
    {
        return [
            'name.required' => 'Please enter your full name.',
            'name.regex' => 'Name can only contain letters, spaces, hyphens, and apostrophes.',
            'email.required' => 'Please enter your email address.',
            'email.email' => 'Please enter a valid email address.',
            'email.unique' => 'This email address is already in use.',
            'phone.regex' => 'Please enter a valid phone number.',
            'website.url' => 'Please enter a valid website URL.',
            'current_password.required_with' => 'Please enter your current password to change it.',
            'current_password.current_password' => 'The current password is incorrect.',
            'password.confirmed' => 'Password confirmation does not match.',
        ];
    }
    
    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Clean phone number
        if ($this->phone) {
            $phone = preg_replace('/[^\d\+\-\(\)\s]/', '', trim($this->phone));
            $this->merge(['phone' => $phone]);
        }
        
        // Clean website URL
        if ($this->website) {
            $website = trim($this->website);
            if (!preg_match('/^https?:\/\//', $website)) {
                $website = 'https://' . $website;
            }
            $this->merge(['website' => $website]);
        }
        
        // Ensure preferences structure
        if (!$this->preferences) {
            $this->merge(['preferences' => []]);
        }
    }
    
    /**
     * Get validated data with cleaned values.
     */
    public function getCleanedData(): array
    {
        $data = $this->validated();
        
        // Remove password fields if not changing password
        if (empty($data['password'])) {
            unset($data['current_password'], $data['password'], $data['password_confirmation']);
        } else {
            // Hash the password
            $data['password'] = bcrypt($data['password']);
            unset($data['current_password'], $data['password_confirmation']);
        }
        
        // Merge preferences with defaults
        if (isset($data['preferences'])) {
            $user = auth()->user();
            $currentPreferences = $user->preferences ?? [];
            $data['preferences'] = array_merge($currentPreferences, $data['preferences']);
        }
        
        return $data;
    }
}