<?php

namespace App\Listeners;

use App\Events\DomainVerified;
use App\Models\TrustSeal;
use App\Models\SecurityCheck;
use App\Services\SecurityMonitoringService;
use App\Notifications\DomainVerifiedNotification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Facades\Log;

class HandleDomainVerification implements ShouldQueue
{
    use InteractsWithQueue;

    protected $securityService;

    /**
     * Create the event listener.
     */
    public function __construct(SecurityMonitoringService $securityService)
    {
        $this->securityService = $securityService;
    }

    /**
     * Handle the event.
     */
    public function handle(DomainVerified $event): void
    {
        $domain = $event->domain;
        
        try {
            // 1. Create trust seal for the domain
            $this->createTrustSeal($domain);
            
            // 2. Run initial security check
            $this->runInitialSecurityCheck($domain);
            
            // 3. Send notification to user
            $this->sendNotification($domain, $event->verificationMethod);
            
            // 4. Log the verification
            $this->logVerification($domain, $event->verificationMethod);
            
            // 5. Update domain statistics
            $this->updateDomainStats($domain);
            
        } catch (\Exception $e) {
            Log::error('Failed to handle domain verification', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Re-throw to trigger retry
            throw $e;
        }
    }
    
    /**
     * Create a trust seal for the verified domain.
     */
    protected function createTrustSeal($domain): void
    {
        if (TrustSeal::where('domain_id', $domain->id)->exists()) {
            return; // Trust seal already exists
        }
        
        TrustSeal::create([
            'domain_id' => $domain->id,
            'is_active' => true,
            'seal_type' => 'basic',
            'customization' => [
                'position' => 'bottom-right',
                'theme' => 'light',
                'size' => 'medium',
                'show_text' => true,
                'text' => 'Verified & Secure',
                'colors' => [
                    'primary' => '#007bff',
                    'secondary' => '#6c757d',
                    'background' => '#ffffff',
                    'text' => '#000000'
                ],
                'animation' => 'pulse',
                'shadow' => true,
                'border_radius' => 8
            ],
            'display_count' => 0,
            'last_displayed_at' => null
        ]);
        
        Log::info('Trust seal created for verified domain', [
            'domain_id' => $domain->id,
            'domain' => $domain->domain
        ]);
    }
    
    /**
     * Run initial security check for the domain.
     */
    protected function runInitialSecurityCheck($domain): void
    {
        try {
            // Run comprehensive security check
            $this->securityService->runSecurityCheck($domain);
            
            Log::info('Initial security check completed for verified domain', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain
            ]);
            
        } catch (\Exception $e) {
            Log::warning('Initial security check failed for verified domain', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'error' => $e->getMessage()
            ]);
            
            // Don't throw - security check failure shouldn't block verification
        }
    }
    
    /**
     * Send notification to the user.
     */
    protected function sendNotification($domain, string $verificationMethod): void
    {
        try {
            $domain->user->notify(new DomainVerifiedNotification($domain, $verificationMethod));
            
            Log::info('Domain verification notification sent', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'user_id' => $domain->user_id,
                'verification_method' => $verificationMethod
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to send domain verification notification', [
                'domain_id' => $domain->id,
                'domain' => $domain->domain,
                'user_id' => $domain->user_id,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Log the verification event.
     */
    protected function logVerification($domain, string $verificationMethod): void
    {
        Log::info('Domain verification completed', [
            'domain_id' => $domain->id,
            'domain' => $domain->domain,
            'user_id' => $domain->user_id,
            'verification_method' => $verificationMethod,
            'verified_at' => now()->toISOString(),
            'user_email' => $domain->user->email,
            'user_account_type' => $domain->user->account_type
        ]);
    }
    
    /**
     * Update domain-related statistics.
     */
    protected function updateDomainStats($domain): void
    {
        try {
            // Update user's domain count cache if needed
            $user = $domain->user;
            $verifiedCount = $user->domains()->where('is_verified', true)->count();
            
            // You could cache this count or trigger other stats updates
            Log::info('Domain stats updated', [
                'user_id' => $user->id,
                'verified_domains_count' => $verifiedCount
            ]);
            
        } catch (\Exception $e) {
            Log::warning('Failed to update domain stats', [
                'domain_id' => $domain->id,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Handle a job failure.
     */
    public function failed(DomainVerified $event, \Throwable $exception): void
    {
        Log::error('Domain verification listener failed', [
            'domain_id' => $event->domain->id,
            'domain' => $event->domain->domain,
            'verification_method' => $event->verificationMethod,
            'error' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);
    }
}