<?php

namespace App\Listeners;

use App\Events\SealViewed;
use App\Models\SealAnalytics;
use App\Services\AnalyticsService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class TrackSealView implements ShouldQueue
{
    use InteractsWithQueue;

    protected $analyticsService;

    /**
     * Create the event listener.
     */
    public function __construct(AnalyticsService $analyticsService)
    {
        $this->analyticsService = $analyticsService;
    }

    /**
     * Handle the event.
     */
    public function handle(SealViewed $event): void
    {
        // Skip tracking if this view shouldn't be tracked (e.g., from bots)
        if (!$event->shouldTrack()) {
            Log::debug('Skipping seal view tracking (bot detected)', [
                'trust_seal_id' => $event->trustSeal->id,
                'user_agent' => $event->viewData['user_agent'] ?? 'unknown'
            ]);
            return;
        }
        
        try {
            // 1. Track the view using AnalyticsService
            $this->trackView($event);
            
            // 2. Update trust seal statistics
            $this->updateTrustSealStats($event);
            
            // 3. Update real-time statistics
            $this->updateRealTimeStats($event);
            
            // 4. Log the view for debugging
            $this->logSealView($event);
            
        } catch (\Exception $e) {
            Log::error('Failed to track seal view', [
                'trust_seal_id' => $event->trustSeal->id,
                'domain_id' => $event->trustSeal->domain_id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Don't re-throw - view tracking failure shouldn't break the response
        }
    }
    
    /**
     * Track the view using AnalyticsService.
     */
    protected function trackView(SealViewed $event): void
    {
        $viewData = array_merge($event->viewData, [
            'is_unique' => $event->isUniqueView(),
            'viewed_at' => $event->viewedAt
        ]);
        
        $this->analyticsService->trackSealView(
            $event->trustSeal,
            $viewData
        );
        
        Log::debug('Seal view tracked via AnalyticsService', [
            'trust_seal_id' => $event->trustSeal->id,
            'is_unique' => $viewData['is_unique']
        ]);
    }
    
    /**
     * Update trust seal display statistics.
     */
    protected function updateTrustSealStats(SealViewed $event): void
    {
        $trustSeal = $event->trustSeal;
        
        // Increment display count
        $trustSeal->increment('display_count');
        
        // Update last displayed timestamp
        $trustSeal->update([
            'last_displayed_at' => $event->viewedAt
        ]);
        
        Log::debug('Trust seal stats updated', [
            'trust_seal_id' => $trustSeal->id,
            'new_display_count' => $trustSeal->display_count + 1
        ]);
    }
    
    /**
     * Update real-time statistics cache.
     */
    protected function updateRealTimeStats(SealViewed $event): void
    {
        $trustSeal = $event->trustSeal;
        $domain = $trustSeal->domain;
        $user = $domain->user;
        
        try {
            // Update real-time view count (last 5 minutes)
            $realtimeKey = "realtime_views_{$user->id}";
            $currentViews = Cache::get($realtimeKey, 0);
            Cache::put($realtimeKey, $currentViews + 1, now()->addMinutes(5));
            
            // Track active countries (last hour)
            $country = $event->viewData['country'] ?? 'Unknown';
            if ($country !== 'Unknown') {
                $countriesKey = "active_countries_{$user->id}";
                $countries = Cache::get($countriesKey, []);
                
                if (!in_array($country, $countries)) {
                    $countries[] = $country;
                    Cache::put($countriesKey, $countries, now()->addHour());
                }
            }
            
            // Track hourly views
            $hourlyKey = "hourly_views_{$user->id}_" . now()->format('Y-m-d-H');
            $hourlyViews = Cache::get($hourlyKey, 0);
            Cache::put($hourlyKey, $hourlyViews + 1, now()->addHours(25)); // Keep for 25 hours
            
            Log::debug('Real-time stats updated', [
                'user_id' => $user->id,
                'trust_seal_id' => $trustSeal->id,
                'realtime_views' => $currentViews + 1,
                'active_countries_count' => count($countries ?? []),
                'hourly_views' => $hourlyViews + 1
            ]);
            
        } catch (\Exception $e) {
            Log::warning('Failed to update real-time stats', [
                'trust_seal_id' => $trustSeal->id,
                'error' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * Log the seal view for debugging and monitoring.
     */
    protected function logSealView(SealViewed $event): void
    {
        $trustSeal = $event->trustSeal;
        $viewData = $event->viewData;
        
        Log::info('Seal view processed', [
            'trust_seal_id' => $trustSeal->id,
            'domain_id' => $trustSeal->domain_id,
            'domain' => $trustSeal->domain->domain,
            'user_id' => $trustSeal->domain->user_id,
            'ip_address' => $this->maskIp($viewData['ip_address'] ?? 'unknown'),
            'country' => $viewData['country'] ?? 'Unknown',
            'browser' => $viewData['browser'] ?? 'Unknown',
            'device_type' => $viewData['device_type'] ?? 'Unknown',
            'referrer_domain' => $event->getReferrerDomain(),
            'is_unique' => $event->isUniqueView(),
            'viewed_at' => $event->viewedAt->toISOString()
        ]);
    }
    
    /**
     * Mask IP address for privacy (keep first 3 octets for IPv4).
     */
    protected function maskIp(string $ip): string
    {
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $parts = explode('.', $ip);
            return implode('.', array_slice($parts, 0, 3)) . '.xxx';
        }
        
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            $parts = explode(':', $ip);
            return implode(':', array_slice($parts, 0, 4)) . '::xxxx';
        }
        
        return 'masked';
    }
    
    /**
     * Handle a job failure.
     */
    public function failed(SealViewed $event, \Throwable $exception): void
    {
        Log::error('Seal view tracking listener failed', [
            'trust_seal_id' => $event->trustSeal->id,
            'domain_id' => $event->trustSeal->domain_id,
            'domain' => $event->trustSeal->domain->domain,
            'user_id' => $event->trustSeal->domain->user_id,
            'error' => $exception->getMessage(),
            'trace' => $exception->getTraceAsString()
        ]);
    }
    
    /**
     * Determine the number of times the job may be attempted.
     */
    public function tries(): int
    {
        return 3;
    }
    
    /**
     * Calculate the number of seconds to wait before retrying the job.
     */
    public function backoff(): array
    {
        return [1, 5, 10];
    }
}