<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Domain extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'domain',
        'status',
        'verified_at',
        'verification_methods',
        'verification_token',
        'ssl_info',
        'security_score',
        'last_security_check',
        'is_active'
    ];

    protected $casts = [
        'verified_at' => 'datetime',
        'verification_methods' => 'array',
        'ssl_info' => 'array',
        'last_security_check' => 'datetime',
        'is_active' => 'boolean',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function verifications(): HasMany
    {
        return $this->hasMany(DomainVerification::class);
    }

    public function trustSeals(): HasMany
    {
        return $this->hasMany(TrustSeal::class);
    }

    public function securityChecks(): HasMany
    {
        return $this->hasMany(SecurityCheck::class);
    }

    public function activeTrustSeal(): HasOne
    {
        return $this->hasOne(TrustSeal::class)->where('is_active', true);
    }

    public function activeSecuritySeal(): HasOne
    {
        return $this->hasOne(TrustSeal::class)
            ->where('is_active', true)
            ->where('seal_type', 'security');
    }

    public function isVerified(): bool
    {
        return $this->status === 'verified';
    }

    public function hasValidSSL(): bool
    {
        return !empty($this->ssl_info) && 
               ($this->ssl_info['valid'] ?? false) && 
               ($this->ssl_info['expires_at'] ?? null) > now();
    }

    public function canHaveSecuritySeal(): bool
    {
        return $this->isVerified() && 
               $this->hasValidSSL() && 
               $this->security_score >= 70;
    }

    public function getSecurityScoreColorAttribute(): string
    {
        if ($this->security_score >= 80) return 'success';
        if ($this->security_score >= 60) return 'warning';
        return 'danger';
    }

    public function getCleanDomainAttribute(): string
    {
        return str_replace(['http://', 'https://', 'www.'], '', $this->domain);
    }
}