<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class SecurityAlert extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'domain_id',
        'alert_type',
        'alert_level',
        'message',
        'details',
        'resolved',
        'resolved_at',
        'resolved_by',
        'alerted_at',
        'source',
        'reference_id',
        'metadata'
    ];

    protected $casts = [
        'details' => 'array',
        'metadata' => 'array',
        'resolved' => 'boolean',
        'alerted_at' => 'datetime',
        'resolved_at' => 'datetime',
    ];

    protected $dates = [
        'alerted_at',
        'resolved_at',
        'deleted_at',
    ];

    /**
     * Get the domain that owns the security alert.
     */
    public function domain(): BelongsTo
    {
        return $this->belongsTo(Domain::class);
    }

    /**
     * Get the user that resolved the alert.
     */
    public function resolver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'resolved_by');
    }

    /**
     * Scope a query to only include unresolved alerts.
     */
    public function scopeUnresolved($query)
    {
        return $query->where('resolved', false);
    }

    /**
     * Scope a query to only include resolved alerts.
     */
    public function scopeResolved($query)
    {
        return $query->where('resolved', true);
    }

    /**
     * Scope a query to filter by alert level.
     */
    public function scopeByLevel($query, $level)
    {
        return $query->where('alert_level', $level);
    }

    /**
     * Scope a query to filter by alert type.
     */
    public function scopeByType($query, $type)
    {
        return $query->where('alert_type', $type);
    }

    /**
     * Get recommended actions for this alert.
     */
    public function getRecommendedActions(): array
    {
        return match($this->alert_type) {
            'ssl_expiry' => [
                'Renew your SSL certificate before it expires',
                'Set up automatic SSL renewal',
                'Monitor SSL certificate status regularly'
            ],
            'malware_detected' => [
                'Scan your website for malware immediately',
                'Clean infected files',
                'Update all plugins and themes',
                'Change all passwords',
                'Contact your hosting provider'
            ],
            'security_headers' => [
                'Implement missing security headers',
                'Review your web server configuration',
                'Test security headers implementation'
            ],
            'vulnerability' => [
                'Update affected software immediately',
                'Apply security patches',
                'Review security configurations'
            ],
            'blacklist' => [
                'Request removal from blacklist',
                'Clean your website of any malicious content',
                'Improve your website security'
            ],
            default => [
                'Review the security issue details',
                'Take appropriate corrective action',
                'Monitor for similar issues'
            ]
        };
    }

    /**
     * Mark the alert as resolved.
     */
    public function markAsResolved($userId = null): bool
    {
        $this->resolved = true;
        $this->resolved_at = now();
        if ($userId) {
            $this->resolved_by = $userId;
        }
        return $this->save();
    }

    /**
     * Get the severity level as a number.
     */
    public function getSeverityAttribute(): int
    {
        return match($this->alert_level) {
            'low' => 1,
            'medium' => 2,
            'high' => 3,
            'critical' => 4,
            'error' => 5,
            default => 0
        };
    }

    /**
     * Get the alert priority.
     */
    public function getPriorityAttribute(): string
    {
        return match($this->alert_level) {
            'critical', 'error' => 'urgent',
            'high' => 'high',
            'medium' => 'normal',
            'low' => 'low',
            default => 'normal'
        };
    }

    /**
     * Check if the alert is critical.
     */
    public function isCritical(): bool
    {
        return in_array($this->alert_level, ['critical', 'error']);
    }

    /**
     * Get the alert age in hours.
     */
    public function getAgeInHours(): float
    {
        return $this->alerted_at->diffInHours(now());
    }
}