<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class SecurityCheck extends Model
{
    use HasFactory;

    protected $fillable = [
        'domain_id',
        'check_type',
        'status',
        'score',
        'details',
        'message',
        'checked_at'
    ];

    protected $casts = [
        'details' => 'array',
        'checked_at' => 'datetime',
    ];

    public function domain(): BelongsTo
    {
        return $this->belongsTo(Domain::class);
    }

    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'pass' => 'success',
            'warning' => 'warning',
            'fail' => 'danger',
            'error' => 'dark',
            default => 'secondary'
        };
    }

    public function getCheckTypeDisplayAttribute(): string
    {
        return match($this->check_type) {
            'ssl' => 'SSL Certificate',
            'malware' => 'Malware Scan',
            'spam' => 'Spam Database',
            'dns' => 'DNS Health',
            'general' => 'General Security',
            default => ucfirst($this->check_type)
        };
    }

    public function getScoreColorAttribute(): string
    {
        if ($this->score >= 80) return 'success';
        if ($this->score >= 60) return 'warning';
        return 'danger';
    }

    public function isPassing(): bool
    {
        return $this->status === 'pass';
    }

    public function isFailing(): bool
    {
        return $this->status === 'fail';
    }

    public function hasWarning(): bool
    {
        return $this->status === 'warning';
    }

    public function hasError(): bool
    {
        return $this->status === 'error';
    }

    public function getRecommendations(): array
    {
        $recommendations = [];
        
        switch ($this->check_type) {
            case 'ssl':
                if ($this->isFailing()) {
                    $recommendations[] = 'Install a valid SSL certificate';
                    $recommendations[] = 'Ensure SSL certificate is not expired';
                    $recommendations[] = 'Verify SSL certificate chain is complete';
                }
                break;
                
            case 'malware':
                if ($this->isFailing()) {
                    $recommendations[] = 'Scan your website for malware immediately';
                    $recommendations[] = 'Update all plugins and themes';
                    $recommendations[] = 'Change all passwords';
                    $recommendations[] = 'Consider professional malware removal';
                }
                break;
                
            case 'spam':
                if ($this->isFailing()) {
                    $recommendations[] = 'Check if your domain is blacklisted';
                    $recommendations[] = 'Review and remove any suspicious content';
                    $recommendations[] = 'Request removal from spam databases';
                }
                break;
                
            case 'dns':
                if ($this->isFailing()) {
                    $recommendations[] = 'Check DNS configuration';
                    $recommendations[] = 'Ensure all DNS records are valid';
                    $recommendations[] = 'Consider using a reliable DNS provider';
                }
                break;
        }
        
        return $recommendations;
    }

    public function scopeLatest($query)
    {
        return $query->orderBy('checked_at', 'desc');
    }

    public function scopeByType($query, $type)
    {
        return $query->where('check_type', $type);
    }

    public function scopePassing($query)
    {
        return $query->where('status', 'pass');
    }

    public function scopeFailing($query)
    {
        return $query->whereIn('status', ['fail', 'error']);
    }
}