<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Subscription extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'plan',
        'status',
        'amount',
        'currency',
        'billing_cycle',
        'started_at',
        'expires_at',
        'cancelled_at',
        'payment_method',
        'payment_reference',
        'features'
    ];

    protected $casts = [
        'started_at' => 'datetime',
        'expires_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'features' => 'array',
        'amount' => 'decimal:2',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function isActive(): bool
    {
        return $this->status === 'active' && 
               $this->expires_at > now() && 
               is_null($this->cancelled_at);
    }

    public function isExpired(): bool
    {
        return $this->expires_at <= now();
    }

    public function isCancelled(): bool
    {
        return !is_null($this->cancelled_at);
    }

    public function isPending(): bool
    {
        return $this->status === 'pending';
    }

    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'active' => 'success',
            'pending' => 'warning',
            'cancelled' => 'danger',
            'expired' => 'dark',
            default => 'secondary'
        };
    }

    public function getPlanDisplayAttribute(): string
    {
        return match($this->plan) {
            'basic' => 'Basic',
            'premium' => 'Premium',
            'enterprise' => 'Enterprise',
            default => ucfirst($this->plan)
        };
    }

    public function getBillingCycleDisplayAttribute(): string
    {
        return match($this->billing_cycle) {
            'monthly' => 'Monthly',
            'yearly' => 'Yearly',
            'lifetime' => 'Lifetime',
            default => ucfirst($this->billing_cycle)
        };
    }

    public function getFormattedAmountAttribute(): string
    {
        return strtoupper($this->currency) . ' ' . number_format($this->amount, 2);
    }

    public function getDaysUntilExpiryAttribute(): int
    {
        return max(0, now()->diffInDays($this->expires_at, false));
    }

    public function getNextBillingDateAttribute(): ?\Carbon\Carbon
    {
        if ($this->isCancelled() || $this->billing_cycle === 'lifetime') {
            return null;
        }
        
        return $this->expires_at;
    }

    public function hasFeature(string $feature): bool
    {
        return in_array($feature, $this->features ?? []);
    }

    public function getAvailableFeatures(): array
    {
        return match($this->plan) {
            'basic' => [
                'trust_seals',
                'basic_analytics',
                'email_support',
                'domains_limit_5'
            ],
            'premium' => [
                'trust_seals',
                'security_seals',
                'advanced_analytics',
                'custom_branding',
                'api_access',
                'priority_support',
                'domains_limit_25',
                'custom_verification'
            ],
            'enterprise' => [
                'trust_seals',
                'security_seals',
                'advanced_analytics',
                'custom_branding',
                'white_label',
                'api_access',
                'dedicated_support',
                'domains_unlimited',
                'custom_verification',
                'advanced_monitoring',
                'custom_integrations'
            ],
            default => []
        };
    }

    public function getFeatureDisplayNames(): array
    {
        return [
            'trust_seals' => 'Trust Seals',
            'security_seals' => 'Security Seals',
            'basic_analytics' => 'Basic Analytics',
            'advanced_analytics' => 'Advanced Analytics',
            'custom_branding' => 'Custom Branding',
            'white_label' => 'White Label Options',
            'api_access' => 'API Access',
            'email_support' => 'Email Support',
            'priority_support' => 'Priority Support',
            'dedicated_support' => 'Dedicated Support',
            'domains_limit_5' => 'Up to 5 Domains',
            'domains_limit_25' => 'Up to 25 Domains',
            'domains_unlimited' => 'Unlimited Domains',
            'custom_verification' => 'Custom Verification Methods',
            'advanced_monitoring' => 'Advanced Security Monitoring',
            'custom_integrations' => 'Custom Integrations'
        ];
    }

    public function canAddDomain(): bool
    {
        $domainCount = $this->user->domains()->count();
        
        return match($this->plan) {
            'basic' => $domainCount < 5,
            'premium' => $domainCount < 25,
            'enterprise' => true,
            default => $domainCount < 1
        };
    }

    public function getDomainLimitAttribute(): int
    {
        return match($this->plan) {
            'basic' => 5,
            'premium' => 25,
            'enterprise' => -1, // unlimited
            default => 1
        };
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active')
                    ->where('expires_at', '>', now())
                    ->whereNull('cancelled_at');
    }

    public function scopeExpiringSoon($query, $days = 7)
    {
        return $query->where('status', 'active')
                    ->whereBetween('expires_at', [now(), now()->addDays($days)])
                    ->whereNull('cancelled_at');
    }
}