<?php

namespace App\Notifications;

use App\Events\SecurityAlert;
use App\Models\SecurityAlert as SecurityAlertModel;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\URL;

class SecurityAlertNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $securityAlert;
    protected $alertModel;

    /**
     * Create a new notification instance.
     */
    public function __construct(SecurityAlert $securityAlert, SecurityAlertModel $alertModel)
    {
        $this->securityAlert = $securityAlert;
        $this->alertModel = $alertModel;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        $channels = ['database'];
        
        // Check user preferences for email notifications
        $preferences = $notifiable->preferences ?? [];
        $notifications = $preferences['notifications'] ?? [];
        
        // Check if user wants security alert emails
        if ($notifications['security_alerts'] ?? true) {
            $channels[] = 'mail';
        }
        
        // For critical alerts, always send email regardless of preferences
        if ($this->securityAlert->alertLevel === 'error' && !in_array('mail', $channels)) {
            $channels[] = 'mail';
        }
        
        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $domain = $this->securityAlert->domain;
        $domainUrl = URL::route('domains.show', $domain);
        $alertsUrl = URL::route('dashboard') . '?tab=security';
        
        $mailMessage = (new MailMessage)
            ->subject($this->getEmailSubject())
            ->greeting($this->getEmailGreeting())
            ->line($this->securityAlert->message)
            ->line("**Domain:** {$domain->domain}")
            ->line("**Alert Type:** {$this->getAlertTypeName()}")
            ->line("**Severity:** {$this->getSeverityName()}")
            ->line("**Time:** {$this->securityAlert->alertedAt->format('M j, Y \\a\\t g:i A T')}")
            ->line('');
        
        // Add details if available
        if (!empty($this->securityAlert->details)) {
            $mailMessage->line('**Details:**');
            
            foreach ($this->securityAlert->details as $key => $value) {
                if (is_string($value)) {
                    $mailMessage->line("• {$key}: {$value}");
                }
            }
            
            $mailMessage->line('');
        }
        
        // Add recommended actions
        $actions = $this->securityAlert->getRecommendedActions();
        if (!empty($actions)) {
            $mailMessage->line('**Recommended Actions:**');
            
            foreach ($actions as $action) {
                $mailMessage->line("• {$action}");
            }
            
            $mailMessage->line('');
        }
        
        $mailMessage->action('View Domain Security', $domainUrl);
        
        // Add urgency-specific messaging
        if ($this->securityAlert->alertLevel === 'error') {
            $mailMessage->line('⚠️ **This is a critical security alert that requires immediate attention.**');
        } elseif ($this->securityAlert->alertLevel === 'warning') {
            $mailMessage->line('⚠️ **Please review this security concern at your earliest convenience.**');
        }
        
        $mailMessage->line('We continuously monitor your domain\'s security to keep your trust seal valid and your visitors safe.');
        
        return $mailMessage->salutation("Best regards,\nThe Trust Seal Security Team");
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'security_alert',
            'title' => $this->getDatabaseTitle(),
            'message' => $this->securityAlert->message,
            'domain_id' => $this->securityAlert->domain->id,
            'domain' => $this->securityAlert->domain->domain,
            'alert_type' => $this->securityAlert->alertType,
            'alert_level' => $this->securityAlert->alertLevel,
            'alert_id' => $this->alertModel->id,
            'alerted_at' => $this->securityAlert->alertedAt->toISOString(),
            'action_url' => URL::route('domains.show', $this->securityAlert->domain),
            'action_text' => 'View Security Details',
            'icon' => $this->getNotificationIcon(),
            'color' => $this->getNotificationColor(),
            'priority' => $this->getNotificationPriority(),
            'metadata' => [
                'domain_id' => $this->securityAlert->domain->id,
                'alert_type' => $this->securityAlert->alertType,
                'alert_level' => $this->securityAlert->alertLevel,
                'alert_model_id' => $this->alertModel->id,
                'details' => $this->securityAlert->details,
                'recommended_actions' => $this->securityAlert->getRecommendedActions()
            ]
        ];
    }

    /**
     * Get the array representation of the notification.
     */
    public function toArray(object $notifiable): array
    {
        return [
            'domain_id' => $this->securityAlert->domain->id,
            'domain' => $this->securityAlert->domain->domain,
            'alert_type' => $this->securityAlert->alertType,
            'alert_level' => $this->securityAlert->alertLevel,
            'message' => $this->securityAlert->message,
            'details' => $this->securityAlert->details,
            'alerted_at' => $this->securityAlert->alertedAt->toISOString()
        ];
    }
    
    /**
     * Get email subject based on alert level and type.
     */
    protected function getEmailSubject(): string
    {
        $domain = $this->securityAlert->domain->domain;
        $severity = $this->getSeverityName();
        $type = $this->getAlertTypeName();
        
        return "[{$severity}] {$type} Alert for {$domain}";
    }
    
    /**
     * Get email greeting based on alert level.
     */
    protected function getEmailGreeting(): string
    {
        return match($this->securityAlert->alertLevel) {
            'error' => 'URGENT: Security Alert',
            'warning' => 'Security Warning',
            'info' => 'Security Notice',
            default => 'Security Alert'
        };
    }
    
    /**
     * Get database notification title.
     */
    protected function getDatabaseTitle(): string
    {
        return "{$this->getSeverityName()} {$this->getAlertTypeName()} Alert";
    }
    
    /**
     * Get human-readable alert type name.
     */
    protected function getAlertTypeName(): string
    {
        return match($this->securityAlert->alertType) {
            'ssl' => 'SSL Certificate',
            'dns' => 'DNS Configuration',
            'malware' => 'Malware Detection',
            'safe_browsing' => 'Safe Browsing',
            'security_score' => 'Security Score',
            default => ucfirst($this->securityAlert->alertType)
        };
    }
    
    /**
     * Get human-readable severity name.
     */
    protected function getSeverityName(): string
    {
        return match($this->securityAlert->alertLevel) {
            'error' => 'Critical',
            'warning' => 'Warning',
            'info' => 'Info',
            default => ucfirst($this->securityAlert->alertLevel)
        };
    }
    
    /**
     * Get notification icon based on alert level.
     */
    protected function getNotificationIcon(): string
    {
        return match($this->securityAlert->alertLevel) {
            'error' => 'exclamation-triangle',
            'warning' => 'exclamation-circle',
            'info' => 'info-circle',
            default => 'shield'
        };
    }
    
    /**
     * Get notification color based on alert level.
     */
    protected function getNotificationColor(): string
    {
        return match($this->securityAlert->alertLevel) {
            'error' => 'red',
            'warning' => 'yellow',
            'info' => 'blue',
            default => 'gray'
        };
    }
    
    /**
     * Get notification priority based on alert level.
     */
    protected function getNotificationPriority(): string
    {
        return match($this->securityAlert->alertLevel) {
            'error' => 'high',
            'warning' => 'medium',
            'info' => 'low',
            default => 'normal'
        };
    }
    
    /**
     * Determine which queues should be used for which notification channels.
     */
    public function viaQueues(): array
    {
        return [
            'mail' => $this->securityAlert->alertLevel === 'error' ? 'urgent' : 'emails',
            'database' => 'default'
        ];
    }
    
    /**
     * Get the notification tags for organization.
     */
    public function tags(): array
    {
        return [
            'security-alert',
            'alert-level:' . $this->securityAlert->alertLevel,
            'alert-type:' . $this->securityAlert->alertType,
            'user:' . $this->securityAlert->domain->user_id,
            'domain:' . $this->securityAlert->domain->id
        ];
    }
}