<?php

namespace App\Notifications;

use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Facades\URL;

class SubscriptionNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $subscription;
    protected $eventType;
    protected $metadata;

    /**
     * Create a new notification instance.
     */
    public function __construct(Subscription $subscription, string $eventType, array $metadata = [])
    {
        $this->subscription = $subscription;
        $this->eventType = $eventType;
        $this->metadata = $metadata;
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via(object $notifiable): array
    {
        $channels = ['database'];
        
        // Check user preferences for email notifications
        $preferences = $notifiable->preferences ?? [];
        $notifications = $preferences['notifications'] ?? [];
        
        // Check if user wants billing/subscription emails
        if ($notifications['billing'] ?? true) {
            $channels[] = 'mail';
        }
        
        // For critical subscription events, always send email
        if (in_array($this->eventType, ['payment_failed', 'subscription_cancelled', 'subscription_expired'])) {
            if (!in_array('mail', $channels)) {
                $channels[] = 'mail';
            }
        }
        
        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        return match($this->eventType) {
            'payment_succeeded' => $this->paymentSucceededMail($notifiable),
            'payment_failed' => $this->paymentFailedMail($notifiable),
            'subscription_created' => $this->subscriptionCreatedMail($notifiable),
            'subscription_updated' => $this->subscriptionUpdatedMail($notifiable),
            'subscription_cancelled' => $this->subscriptionCancelledMail($notifiable),
            'subscription_expired' => $this->subscriptionExpiredMail($notifiable),
            'subscription_renewed' => $this->subscriptionRenewedMail($notifiable),
            'trial_ending' => $this->trialEndingMail($notifiable),
            'upgrade_success' => $this->upgradeSuccessMail($notifiable),
            default => $this->genericSubscriptionMail($notifiable)
        };
    }

    /**
     * Get the database representation of the notification.
     */
    public function toDatabase(object $notifiable): array
    {
        return [
            'type' => 'subscription_event',
            'event_type' => $this->eventType,
            'title' => $this->getDatabaseTitle(),
            'message' => $this->getDatabaseMessage(),
            'subscription_id' => $this->subscription->id,
            'plan' => $this->subscription->plan,
            'status' => $this->subscription->status,
            'action_url' => $this->getActionUrl(),
            'action_text' => $this->getActionText(),
            'icon' => $this->getNotificationIcon(),
            'color' => $this->getNotificationColor(),
            'priority' => $this->getNotificationPriority(),
            'metadata' => array_merge($this->metadata, [
                'subscription_id' => $this->subscription->id,
                'plan' => $this->subscription->plan,
                'status' => $this->subscription->status,
                'amount' => $this->subscription->amount,
                'billing_cycle' => $this->subscription->billing_cycle,
                'event_timestamp' => now()->toISOString()
            ])
        ];
    }

    /**
     * Payment succeeded email.
     */
    protected function paymentSucceededMail(object $notifiable): MailMessage
    {
        $amount = $this->formatAmount($this->subscription->amount);
        $billingUrl = URL::route('subscription.billing');
        
        return (new MailMessage)
            ->subject('Payment Successful - Trust Seal Service')
            ->greeting('Payment Confirmation')
            ->line("We've successfully processed your payment of {$amount} for your {$this->subscription->plan} plan.")
            ->line("**Subscription Details:**")
            ->line("• Plan: {$this->subscription->plan}")
            ->line("• Billing Cycle: {$this->subscription->billing_cycle}")
            ->line("• Next Billing Date: {$this->subscription->next_billing_date->format('M j, Y')}")
            ->line("• Status: Active")
            ->action('View Billing Details', $billingUrl)
            ->line('Thank you for your continued trust in our service!')
            ->salutation("Best regards,\nThe Trust Seal Team");
    }

    /**
     * Payment failed email.
     */
    protected function paymentFailedMail(object $notifiable): MailMessage
    {
        $amount = $this->formatAmount($this->subscription->amount);
        $billingUrl = URL::route('subscription.billing');
        $reason = $this->metadata['failure_reason'] ?? 'Payment was declined';
        
        return (new MailMessage)
            ->subject('Payment Failed - Action Required')
            ->greeting('Payment Issue')
            ->line("We were unable to process your payment of {$amount} for your {$this->subscription->plan} plan.")
            ->line("**Reason:** {$reason}")
            ->line('**What you need to do:**')
            ->line('• Update your payment method')
            ->line('• Ensure sufficient funds are available')
            ->line('• Contact your bank if the issue persists')
            ->action('Update Payment Method', $billingUrl)
            ->line('⚠️ **Important:** Your service may be suspended if payment is not resolved within 7 days.')
            ->line('If you need assistance, please contact our support team.')
            ->salutation("Best regards,\nThe Trust Seal Team");
    }

    /**
     * Subscription created email.
     */
    protected function subscriptionCreatedMail(object $notifiable): MailMessage
    {
        $dashboardUrl = URL::route('dashboard');
        $billingUrl = URL::route('subscription.billing');
        
        return (new MailMessage)
            ->subject('Welcome to Trust Seal ' . $this->subscription->plan . ' Plan!')
            ->greeting('Welcome!')
            ->line("Thank you for subscribing to our {$this->subscription->plan} plan!")
            ->line('**Your subscription includes:**')
            ->line($this->getPlanFeatures())
            ->line('**Getting Started:**')
            ->line('• Add and verify your domains')
            ->line('• Customize your trust seals')
            ->line('• Embed trust seals on your websites')
            ->line('• Monitor your security and analytics')
            ->action('Go to Dashboard', $dashboardUrl)
            ->line('If you have any questions, our support team is here to help!')
            ->salutation("Welcome aboard,\nThe Trust Seal Team");
    }

    /**
     * Subscription cancelled email.
     */
    protected function subscriptionCancelledMail(object $notifiable): MailMessage
    {
        $expiresAt = $this->subscription->expires_at;
        $billingUrl = URL::route('subscription.billing');
        
        return (new MailMessage)
            ->subject('Subscription Cancelled')
            ->greeting('Subscription Cancelled')
            ->line("Your {$this->subscription->plan} subscription has been cancelled.")
            ->line("**Important Information:**")
            ->line("• Your service will continue until {$expiresAt->format('M j, Y')}")
            ->line('• All your domains and trust seals will remain active until then')
            ->line('• You can reactivate your subscription at any time before it expires')
            ->line('• Your data will be preserved for 30 days after expiration')
            ->action('Manage Subscription', $billingUrl)
            ->line('We\'re sorry to see you go. If there\'s anything we can do to improve our service, please let us know.')
            ->salutation("Best regards,\nThe Trust Seal Team");
    }

    /**
     * Get database title based on event type.
     */
    protected function getDatabaseTitle(): string
    {
        return match($this->eventType) {
            'payment_succeeded' => 'Payment Successful',
            'payment_failed' => 'Payment Failed',
            'subscription_created' => 'Subscription Created',
            'subscription_updated' => 'Subscription Updated',
            'subscription_cancelled' => 'Subscription Cancelled',
            'subscription_expired' => 'Subscription Expired',
            'subscription_renewed' => 'Subscription Renewed',
            'trial_ending' => 'Trial Ending Soon',
            'upgrade_success' => 'Plan Upgraded',
            default => 'Subscription Event'
        };
    }

    /**
     * Get database message based on event type.
     */
    protected function getDatabaseMessage(): string
    {
        return match($this->eventType) {
            'payment_succeeded' => "Payment of {$this->formatAmount($this->subscription->amount)} processed successfully.",
            'payment_failed' => 'Payment failed. Please update your payment method.',
            'subscription_created' => "Welcome to the {$this->subscription->plan} plan!",
            'subscription_updated' => "Your {$this->subscription->plan} subscription has been updated.",
            'subscription_cancelled' => "Your {$this->subscription->plan} subscription has been cancelled.",
            'subscription_expired' => "Your {$this->subscription->plan} subscription has expired.",
            'subscription_renewed' => "Your {$this->subscription->plan} subscription has been renewed.",
            'trial_ending' => 'Your trial period ends soon. Upgrade to continue using premium features.',
            'upgrade_success' => "Successfully upgraded to {$this->subscription->plan} plan.",
            default => 'Subscription event occurred.'
        };
    }

    /**
     * Get action URL based on event type.
     */
    protected function getActionUrl(): string
    {
        return match($this->eventType) {
            'payment_failed' => URL::route('subscription.billing'),
            'trial_ending' => URL::route('subscription.plans'),
            default => URL::route('subscription.manage')
        };
    }

    /**
     * Get action text based on event type.
     */
    protected function getActionText(): string
    {
        return match($this->eventType) {
            'payment_failed' => 'Update Payment Method',
            'trial_ending' => 'Upgrade Now',
            'subscription_created' => 'Go to Dashboard',
            default => 'Manage Subscription'
        };
    }

    /**
     * Get notification icon based on event type.
     */
    protected function getNotificationIcon(): string
    {
        return match($this->eventType) {
            'payment_succeeded', 'subscription_renewed', 'upgrade_success' => 'check-circle',
            'payment_failed' => 'exclamation-triangle',
            'subscription_created' => 'star',
            'subscription_cancelled', 'subscription_expired' => 'x-circle',
            'trial_ending' => 'clock',
            default => 'credit-card'
        };
    }

    /**
     * Get notification color based on event type.
     */
    protected function getNotificationColor(): string
    {
        return match($this->eventType) {
            'payment_succeeded', 'subscription_renewed', 'upgrade_success', 'subscription_created' => 'green',
            'payment_failed', 'subscription_expired' => 'red',
            'subscription_cancelled' => 'orange',
            'trial_ending' => 'yellow',
            default => 'blue'
        };
    }

    /**
     * Get notification priority based on event type.
     */
    protected function getNotificationPriority(): string
    {
        return match($this->eventType) {
            'payment_failed', 'subscription_expired' => 'high',
            'trial_ending', 'subscription_cancelled' => 'medium',
            default => 'normal'
        };
    }

    /**
     * Format amount for display.
     */
    protected function formatAmount(float $amount): string
    {
        return '$' . number_format($amount, 2);
    }

    /**
     * Get plan features description.
     */
    protected function getPlanFeatures(): string
    {
        return match($this->subscription->plan) {
            'Basic' => '• Up to 3 domains\n• Basic trust seals\n• Email support',
            'Professional' => '• Up to 10 domains\n• Advanced trust seals\n• Priority support\n• Advanced analytics',
            'Enterprise' => '• Unlimited domains\n• Custom trust seals\n• Dedicated support\n• Advanced security monitoring\n• API access',
            default => '• Premium features included'
        };
    }

    /**
     * Additional email methods for other event types.
     */
    protected function subscriptionUpdatedMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Subscription Updated')
            ->line("Your {$this->subscription->plan} subscription has been updated successfully.")
            ->action('View Details', URL::route('subscription.manage'));
    }

    protected function subscriptionExpiredMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Subscription Expired')
            ->line("Your {$this->subscription->plan} subscription has expired.")
            ->line('Reactivate your subscription to continue using premium features.')
            ->action('Reactivate', URL::route('subscription.plans'));
    }

    protected function subscriptionRenewedMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Subscription Renewed')
            ->line("Your {$this->subscription->plan} subscription has been renewed successfully.")
            ->action('View Details', URL::route('subscription.manage'));
    }

    protected function trialEndingMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Trial Ending Soon')
            ->line('Your trial period will end soon.')
            ->line('Upgrade now to continue enjoying premium features.')
            ->action('Upgrade Now', URL::route('subscription.plans'));
    }

    protected function upgradeSuccessMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Plan Upgraded Successfully')
            ->line("Congratulations! You've successfully upgraded to the {$this->subscription->plan} plan.")
            ->line('Your new features are now available.')
            ->action('Explore Features', URL::route('dashboard'));
    }

    protected function genericSubscriptionMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Subscription Update')
            ->line("Your {$this->subscription->plan} subscription has been updated.")
            ->action('View Details', URL::route('subscription.manage'));
    }

    /**
     * Get the notification tags for organization.
     */
    public function tags(): array
    {
        return [
            'subscription-event',
            'event-type:' . $this->eventType,
            'plan:' . $this->subscription->plan,
            'user:' . $this->subscription->user_id,
            'subscription:' . $this->subscription->id
        ];
    }

    /**
     * Determine which queues should be used for which notification channels.
     */
    public function viaQueues(): array
    {
        return [
            'mail' => in_array($this->eventType, ['payment_failed', 'subscription_expired']) ? 'urgent' : 'emails',
            'database' => 'default'
        ];
    }
}