<?php

namespace App\Services;

use App\Models\TrustSeal;
use App\Models\SealAnalytics;
use Illuminate\Support\Facades\Request;
use Illuminate\Http\Request as HttpRequest;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Jenssegers\Agent\Agent;

class AnalyticsService
{
    /**
     * Track seal view
     */
    public function trackSealView(TrustSeal $seal, HttpRequest $request = null): void
    {
        $request = $request ?? request();
        $agent = new Agent();
        $agent->setUserAgent($request->userAgent());
        
        // Get location data (simplified - in production you'd use a GeoIP service)
        $ipAddress = $request->ip();
        $location = $this->getLocationFromIP($ipAddress);
        
        SealAnalytics::create([
            'trust_seal_id' => $seal->id,
            'ip_address' => $ipAddress,
            'user_agent' => $request->userAgent(),
            'referrer' => $request->header('referer'),
            'country_code' => $location['country_code'] ?? null,
            'city' => $location['city'] ?? null,
            'browser' => $agent->browser(),
            'os' => $agent->platform(),
            'device_type' => $this->getDeviceType($agent),
            'viewed_at' => now()
        ]);
        
        // Update seal display count
        $seal->increment('display_count');
        
        // Update first and last displayed timestamps
        if (!$seal->first_displayed_at) {
            $seal->update(['first_displayed_at' => now()]);
        }
        $seal->update(['last_displayed_at' => now()]);
    }
    
    /**
     * Get analytics data for domain
     */
    public function getDomainAnalytics(int $domainId, string $period = '30d'): array
    {
        $seals = TrustSeal::where('domain_id', $domainId)->pluck('id');
        
        if ($seals->isEmpty()) {
            return $this->getEmptyAnalytics();
        }
        
        $dateRange = $this->getDateRange($period);
        
        return [
            'total_views' => $this->getTotalViews($seals, $dateRange),
            'unique_visitors' => $this->getUniqueVisitors($seals, $dateRange),
            'daily_views' => $this->getDailyViews($seals, $dateRange),
            'top_countries' => $this->getTopCountries($seals, $dateRange),
            'top_browsers' => $this->getTopBrowsers($seals, $dateRange),
            'device_stats' => $this->getDeviceStats($seals, $dateRange),
            'referrer_stats' => $this->getReferrerStats($seals, $dateRange),
            'hourly_distribution' => $this->getHourlyDistribution($seals, $dateRange)
        ];
    }
    
    /**
     * Get analytics for user (all domains)
     */
    public function getUserAnalytics(int $userId, string $period = '30d'): array
    {
        $domainIds = \App\Models\Domain::where('user_id', $userId)->pluck('id');
        $seals = TrustSeal::whereIn('domain_id', $domainIds)->pluck('id');
        
        if ($seals->isEmpty()) {
            return $this->getEmptyAnalytics();
        }
        
        $dateRange = $this->getDateRange($period);
        
        return [
            'total_views' => $this->getTotalViews($seals, $dateRange),
            'unique_visitors' => $this->getUniqueVisitors($seals, $dateRange),
            'daily_views' => $this->getDailyViews($seals, $dateRange),
            'top_countries' => $this->getTopCountries($seals, $dateRange),
            'top_browsers' => $this->getTopBrowsers($seals, $dateRange),
            'device_stats' => $this->getDeviceStats($seals, $dateRange),
            'domain_performance' => $this->getDomainPerformance($userId, $dateRange),
            'growth_metrics' => $this->getGrowthMetrics($seals, $dateRange)
        ];
    }
    
    /**
     * Get total views
     */
    private function getTotalViews($sealIds, array $dateRange): int
    {
        return SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->count();
    }
    
    /**
     * Get unique visitors
     */
    private function getUniqueVisitors($sealIds, array $dateRange): int
    {
        return SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->distinct('ip_address')
            ->count('ip_address');
    }
    
    /**
     * Get daily views
     */
    private function getDailyViews($sealIds, array $dateRange): array
    {
        $results = SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->select(DB::raw('DATE(viewed_at) as date'), DB::raw('COUNT(*) as views'))
            ->groupBy('date')
            ->orderBy('date')
            ->get();
        
        $dailyViews = [];
        $current = Carbon::parse($dateRange[0]);
        $end = Carbon::parse($dateRange[1]);
        
        while ($current <= $end) {
            $dateStr = $current->format('Y-m-d');
            $views = $results->firstWhere('date', $dateStr)?->views ?? 0;
            
            $dailyViews[] = [
                'date' => $dateStr,
                'views' => $views,
                'formatted_date' => $current->format('M d')
            ];
            
            $current->addDay();
        }
        
        return $dailyViews;
    }
    
    /**
     * Get top countries
     */
    private function getTopCountries($sealIds, array $dateRange, int $limit = 10): array
    {
        return SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->whereNotNull('country_code')
            ->select('country_code', DB::raw('COUNT(*) as views'))
            ->groupBy('country_code')
            ->orderByDesc('views')
            ->limit($limit)
            ->get()
            ->map(function ($item) {
                return [
                    'country_code' => $item->country_code,
                    'country_name' => $this->getCountryName($item->country_code),
                    'views' => $item->views
                ];
            })
            ->toArray();
    }
    
    /**
     * Get top browsers
     */
    private function getTopBrowsers($sealIds, array $dateRange, int $limit = 10): array
    {
        return SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->whereNotNull('browser')
            ->select('browser', DB::raw('COUNT(*) as views'))
            ->groupBy('browser')
            ->orderByDesc('views')
            ->limit($limit)
            ->get()
            ->toArray();
    }
    
    /**
     * Get device statistics
     */
    private function getDeviceStats($sealIds, array $dateRange): array
    {
        $stats = SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->select('device_type', DB::raw('COUNT(*) as views'))
            ->groupBy('device_type')
            ->get();
        
        $total = $stats->sum('views');
        
        return $stats->map(function ($item) use ($total) {
            return [
                'device_type' => $item->device_type,
                'views' => $item->views,
                'percentage' => $total > 0 ? round(($item->views / $total) * 100, 1) : 0
            ];
        })->toArray();
    }
    
    /**
     * Get referrer statistics
     */
    private function getReferrerStats($sealIds, array $dateRange, int $limit = 10): array
    {
        return SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->whereNotNull('referrer')
            ->select('referrer', DB::raw('COUNT(*) as views'))
            ->groupBy('referrer')
            ->orderByDesc('views')
            ->limit($limit)
            ->get()
            ->map(function ($item) {
                return [
                    'referrer' => $item->referrer,
                    'domain' => parse_url($item->referrer, PHP_URL_HOST) ?? 'Unknown',
                    'views' => $item->views
                ];
            })
            ->toArray();
    }
    
    /**
     * Get hourly distribution
     */
    private function getHourlyDistribution($sealIds, array $dateRange): array
    {
        $results = SealAnalytics::whereIn('trust_seal_id', $sealIds)
            ->whereBetween('viewed_at', $dateRange)
            ->select(DB::raw('HOUR(viewed_at) as hour'), DB::raw('COUNT(*) as views'))
            ->groupBy('hour')
            ->orderBy('hour')
            ->get();
        
        $hourlyData = [];
        for ($hour = 0; $hour < 24; $hour++) {
            $views = $results->firstWhere('hour', $hour)?->views ?? 0;
            $hourlyData[] = [
                'hour' => $hour,
                'formatted_hour' => sprintf('%02d:00', $hour),
                'views' => $views
            ];
        }
        
        return $hourlyData;
    }
    
    /**
     * Get domain performance comparison
     */
    private function getDomainPerformance(int $userId, array $dateRange): array
    {
        $domains = \App\Models\Domain::where('user_id', $userId)
            ->with(['trustSeals'])
            ->get();
        
        $performance = [];
        
        foreach ($domains as $domain) {
            $sealIds = $domain->trustSeals->pluck('id');
            
            if ($sealIds->isNotEmpty()) {
                $views = $this->getTotalViews($sealIds, $dateRange);
                $uniqueVisitors = $this->getUniqueVisitors($sealIds, $dateRange);
                
                $performance[] = [
                    'domain' => $domain->domain,
                    'views' => $views,
                    'unique_visitors' => $uniqueVisitors,
                    'security_score' => $domain->security_score
                ];
            }
        }
        
        // Sort by views descending
        usort($performance, function ($a, $b) {
            return $b['views'] - $a['views'];
        });
        
        return $performance;
    }
    
    /**
     * Get growth metrics
     */
    private function getGrowthMetrics($sealIds, array $dateRange): array
    {
        $midPoint = Carbon::parse($dateRange[0])->addDays(
            Carbon::parse($dateRange[0])->diffInDays(Carbon::parse($dateRange[1])) / 2
        );
        
        $firstHalf = [$dateRange[0], $midPoint->format('Y-m-d')];
        $secondHalf = [$midPoint->addDay()->format('Y-m-d'), $dateRange[1]];
        
        $firstHalfViews = $this->getTotalViews($sealIds, $firstHalf);
        $secondHalfViews = $this->getTotalViews($sealIds, $secondHalf);
        
        $growthRate = 0;
        if ($firstHalfViews > 0) {
            $growthRate = round((($secondHalfViews - $firstHalfViews) / $firstHalfViews) * 100, 1);
        }
        
        return [
            'first_half_views' => $firstHalfViews,
            'second_half_views' => $secondHalfViews,
            'growth_rate' => $growthRate,
            'trend' => $growthRate > 0 ? 'up' : ($growthRate < 0 ? 'down' : 'stable')
        ];
    }
    
    /**
     * Get date range for period
     */
    private function getDateRange(string $period): array
    {
        $end = now();
        
        switch ($period) {
            case '7d':
                $start = $end->copy()->subDays(7);
                break;
            case '30d':
                $start = $end->copy()->subDays(30);
                break;
            case '90d':
                $start = $end->copy()->subDays(90);
                break;
            case '1y':
                $start = $end->copy()->subYear();
                break;
            default:
                $start = $end->copy()->subDays(30);
        }
        
        return [$start->format('Y-m-d'), $end->format('Y-m-d')];
    }
    
    /**
     * Get empty analytics structure
     */
    private function getEmptyAnalytics(): array
    {
        return [
            'total_views' => 0,
            'unique_visitors' => 0,
            'daily_views' => [],
            'top_countries' => [],
            'top_browsers' => [],
            'device_stats' => [],
            'referrer_stats' => [],
            'hourly_distribution' => []
        ];
    }
    
    /**
     * Get device type from agent
     */
    private function getDeviceType(Agent $agent): string
    {
        if ($agent->isMobile()) {
            return 'mobile';
        } elseif ($agent->isTablet()) {
            return 'tablet';
        } else {
            return 'desktop';
        }
    }
    
    /**
     * Get location from IP (simplified - use real GeoIP service in production)
     */
    private function getLocationFromIP(string $ip): array
    {
        // In production, integrate with a GeoIP service like MaxMind, IPGeolocation, etc.
        // For now, return default values
        return [
            'country_code' => null,
            'city' => null
        ];
    }
    
    /**
     * Get country name from country code
     */
    private function getCountryName(string $countryCode): string
    {
        $countries = [
            'US' => 'United States',
            'GB' => 'United Kingdom',
            'CA' => 'Canada',
            'AU' => 'Australia',
            'DE' => 'Germany',
            'FR' => 'France',
            'IT' => 'Italy',
            'ES' => 'Spain',
            'NL' => 'Netherlands',
            'SE' => 'Sweden',
            'NO' => 'Norway',
            'DK' => 'Denmark',
            'FI' => 'Finland',
            'JP' => 'Japan',
            'KR' => 'South Korea',
            'CN' => 'China',
            'IN' => 'India',
            'BR' => 'Brazil',
            'MX' => 'Mexico',
            'AR' => 'Argentina'
        ];
        
        return $countries[$countryCode] ?? $countryCode;
    }
    
    /**
     * Export analytics data to CSV
     */
    public function exportAnalytics(int $userId, string $period = '30d'): string
    {
        $analytics = $this->getUserAnalytics($userId, $period);
        
        $csvData = [];
        $csvData[] = ['Metric', 'Value'];
        $csvData[] = ['Total Views', $analytics['total_views']];
        $csvData[] = ['Unique Visitors', $analytics['unique_visitors']];
        $csvData[] = [''];
        $csvData[] = ['Daily Views'];
        $csvData[] = ['Date', 'Views'];
        
        foreach ($analytics['daily_views'] as $day) {
            $csvData[] = [$day['date'], $day['views']];
        }
        
        $csvData[] = [''];
        $csvData[] = ['Top Countries'];
        $csvData[] = ['Country', 'Views'];
        
        foreach ($analytics['top_countries'] as $country) {
            $csvData[] = [$country['country_name'], $country['views']];
        }
        
        // Convert to CSV string
        $output = '';
        foreach ($csvData as $row) {
            $output .= implode(',', array_map(function($field) {
                return '"' . str_replace('"', '""', $field) . '"';
            }, $row)) . "\n";
        }
        
        return $output;
    }
    
    /**
     * Get real-time statistics
     */
    public function getRealTimeStats(int $userId): array
    {
        $domainIds = \App\Models\Domain::where('user_id', $userId)->pluck('id');
        $sealIds = TrustSeal::whereIn('domain_id', $domainIds)->pluck('id');
        
        if ($sealIds->isEmpty()) {
            return [
                'views_last_hour' => 0,
                'views_today' => 0,
                'active_visitors' => 0,
                'recent_countries' => []
            ];
        }
        
        return [
            'views_last_hour' => SealAnalytics::whereIn('trust_seal_id', $sealIds)
                ->where('viewed_at', '>', now()->subHour())
                ->count(),
            'views_today' => SealAnalytics::whereIn('trust_seal_id', $sealIds)
                ->whereDate('viewed_at', today())
                ->count(),
            'active_visitors' => SealAnalytics::whereIn('trust_seal_id', $sealIds)
                ->where('viewed_at', '>', now()->subMinutes(5))
                ->distinct('ip_address')
                ->count('ip_address'),
            'recent_countries' => SealAnalytics::whereIn('trust_seal_id', $sealIds)
                ->where('viewed_at', '>', now()->subHour())
                ->whereNotNull('country_code')
                ->select('country_code', DB::raw('COUNT(*) as views'))
                ->groupBy('country_code')
                ->orderByDesc('views')
                ->limit(5)
                ->get()
                ->map(function ($item) {
                    return [
                        'country_code' => $item->country_code,
                        'country_name' => $this->getCountryName($item->country_code),
                        'views' => $item->views
                    ];
                })
                ->toArray()
        ];
    }
}