<?php

namespace App\Services;

use App\Models\Domain;
use App\Models\SecurityCheck;
use App\Models\SecurityAlert;
use App\Events\SecurityAlert as SecurityAlertEvent;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;

class SecurityCheckService
{
    protected $sslConfig;
    protected $malwareConfig;
    protected $blacklistConfig;
    protected $headerConfig;
    protected $vulnerabilityConfig;

    public function __construct()
    {
        $this->sslConfig = config('security.ssl_monitoring', []);
        $this->malwareConfig = config('security.malware_scanning', []);
        $this->blacklistConfig = config('security.blacklist_monitoring', []);
        $this->headerConfig = config('security.headers', []);
        $this->vulnerabilityConfig = config('security.vulnerability_scanning', []);
    }

    /**
     * Run comprehensive security check for a domain.
     */
    public function runSecurityCheck(Domain $domain): SecurityCheck
    {
        Log::info('Starting security check for domain', ['domain' => $domain->domain_name]);

        $securityCheck = SecurityCheck::create([
            'domain_id' => $domain->id,
            'status' => 'running',
            'started_at' => now(),
            'ssl_status' => 'pending',
            'malware_status' => 'pending',
            'blacklist_status' => 'pending',
            'headers_status' => 'pending',
            'vulnerability_status' => 'pending'
        ]);

        try {
            // Run all security checks
            $this->checkSslCertificate($domain, $securityCheck);
            $this->checkMalware($domain, $securityCheck);
            $this->checkBlacklists($domain, $securityCheck);
            $this->checkSecurityHeaders($domain, $securityCheck);
            $this->checkVulnerabilities($domain, $securityCheck);

            // Calculate overall score
            $this->calculateSecurityScore($securityCheck);

            $securityCheck->update([
                'status' => 'completed',
                'completed_at' => now()
            ]);

            Log::info('Security check completed', [
                'domain' => $domain->domain_name,
                'score' => $securityCheck->security_score
            ]);

        } catch (\Exception $e) {
            Log::error('Security check failed', [
                'domain' => $domain->domain_name,
                'error' => $e->getMessage()
            ]);

            $securityCheck->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
                'completed_at' => now()
            ]);
        }

        return $securityCheck;
    }

    /**
     * Check SSL certificate status and configuration.
     */
    protected function checkSslCertificate(Domain $domain, SecurityCheck $securityCheck): void
    {
        try {
            $url = 'https://' . $domain->domain_name;
            $context = stream_context_create([
                'ssl' => [
                    'capture_peer_cert' => true,
                    'verify_peer' => false,
                    'verify_peer_name' => false
                ]
            ]);

            $stream = @stream_socket_client(
                'ssl://' . $domain->domain_name . ':443',
                $errno,
                $errstr,
                30,
                STREAM_CLIENT_CONNECT,
                $context
            );

            if (!$stream) {
                throw new \Exception("Cannot connect to {$domain->domain_name}:443 - {$errstr}");
            }

            $params = stream_context_get_params($stream);
            $cert = $params['options']['ssl']['peer_certificate'];
            $certData = openssl_x509_parse($cert);

            $expiryDate = Carbon::createFromTimestamp($certData['validTo_time_t']);
            $issueDate = Carbon::createFromTimestamp($certData['validFrom_time_t']);
            $daysUntilExpiry = now()->diffInDays($expiryDate, false);

            $sslData = [
                'has_ssl' => true,
                'issuer' => $certData['issuer']['CN'] ?? 'Unknown',
                'subject' => $certData['subject']['CN'] ?? 'Unknown',
                'valid_from' => $issueDate->toDateTimeString(),
                'valid_to' => $expiryDate->toDateTimeString(),
                'days_until_expiry' => $daysUntilExpiry,
                'is_valid' => $daysUntilExpiry > 0,
                'is_self_signed' => $this->isSelfSigned($certData),
                'signature_algorithm' => $certData['signatureTypeSN'] ?? 'Unknown'
            ];

            // Check for SSL issues
            if ($daysUntilExpiry <= 30) {
                $this->createSecurityAlert($domain, 'ssl_expiring', 'SSL certificate expires soon', [
                    'days_until_expiry' => $daysUntilExpiry,
                    'expiry_date' => $expiryDate->toDateTimeString()
                ]);
            }

            if ($sslData['is_self_signed']) {
                $this->createSecurityAlert($domain, 'ssl_self_signed', 'SSL certificate is self-signed', $sslData);
            }

            $securityCheck->update([
                'ssl_status' => 'passed',
                'ssl_data' => $sslData
            ]);

            fclose($stream);

        } catch (\Exception $e) {
            $securityCheck->update([
                'ssl_status' => 'failed',
                'ssl_data' => ['error' => $e->getMessage()]
            ]);

            $this->createSecurityAlert($domain, 'ssl_error', 'SSL certificate check failed', [
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Check for malware using multiple scanning services.
     */
    protected function checkMalware(Domain $domain, SecurityCheck $securityCheck): void
    {
        $results = [];
        $overallStatus = 'clean';

        // Google Safe Browsing
        if (config('security.malware_scanning.google_safe_browsing.enabled')) {
            $results['google_safe_browsing'] = $this->checkGoogleSafeBrowsing($domain);
            if ($results['google_safe_browsing']['threat_detected']) {
                $overallStatus = 'infected';
            }
        }

        // VirusTotal
        if (config('security.malware_scanning.virustotal.enabled')) {
            $results['virustotal'] = $this->checkVirusTotal($domain);
            if ($results['virustotal']['threat_detected']) {
                $overallStatus = 'infected';
            }
        }

        // URLVoid
        if (config('security.malware_scanning.urlvoid.enabled')) {
            $results['urlvoid'] = $this->checkUrlVoid($domain);
            if ($results['urlvoid']['threat_detected']) {
                $overallStatus = 'infected';
            }
        }

        $securityCheck->update([
            'malware_status' => $overallStatus === 'clean' ? 'passed' : 'failed',
            'malware_data' => $results
        ]);

        if ($overallStatus === 'infected') {
            $this->createSecurityAlert($domain, 'malware_detected', 'Malware detected on domain', $results);
        }
    }

    /**
     * Check security headers.
     */
    protected function checkSecurityHeaders(Domain $domain, SecurityCheck $securityCheck): void
    {
        try {
            $response = Http::timeout(30)->get('https://' . $domain->domain_name);
            $headers = $response->headers();

            $requiredHeaders = [
                'X-Frame-Options',
                'X-Content-Type-Options',
                'X-XSS-Protection',
                'Strict-Transport-Security',
                'Content-Security-Policy',
                'Referrer-Policy'
            ];

            $headerResults = [];
            $missingHeaders = [];

            foreach ($requiredHeaders as $header) {
                $headerKey = strtolower($header);
                $headerValue = $headers[$headerKey][0] ?? null;
                
                $headerResults[$header] = [
                    'present' => !empty($headerValue),
                    'value' => $headerValue
                ];

                if (empty($headerValue)) {
                    $missingHeaders[] = $header;
                }
            }

            $score = (count($requiredHeaders) - count($missingHeaders)) / count($requiredHeaders) * 100;

            $securityCheck->update([
                'headers_status' => $score >= 70 ? 'passed' : 'failed',
                'headers_data' => [
                    'headers' => $headerResults,
                    'missing_headers' => $missingHeaders,
                    'score' => $score
                ]
            ]);

            if (!empty($missingHeaders)) {
                $this->createSecurityAlert($domain, 'missing_security_headers', 'Missing security headers', [
                    'missing_headers' => $missingHeaders,
                    'score' => $score
                ]);
            }

        } catch (\Exception $e) {
            $securityCheck->update([
                'headers_status' => 'failed',
                'headers_data' => ['error' => $e->getMessage()]
            ]);
        }
    }

    /**
     * Check blacklist status.
     */
    protected function checkBlacklists(Domain $domain, SecurityCheck $securityCheck): void
    {
        $results = [];
        $isBlacklisted = false;

        // Check multiple blacklist services
        $blacklists = [
            'google_safe_browsing' => 'Google Safe Browsing',
            'malware_domain_list' => 'Malware Domain List',
            'phishtank' => 'PhishTank',
            'openphish' => 'OpenPhish'
        ];

        foreach ($blacklists as $service => $name) {
            try {
                $results[$service] = $this->checkBlacklistService($domain, $service);
                if ($results[$service]['blacklisted']) {
                    $isBlacklisted = true;
                }
            } catch (\Exception $e) {
                $results[$service] = [
                    'blacklisted' => false,
                    'error' => $e->getMessage()
                ];
            }
        }

        $securityCheck->update([
            'blacklist_status' => $isBlacklisted ? 'failed' : 'passed',
            'blacklist_data' => $results
        ]);

        if ($isBlacklisted) {
            $this->createSecurityAlert($domain, 'domain_blacklisted', 'Domain found on blacklists', $results);
        }
    }

    /**
     * Check for known vulnerabilities.
     */
    protected function checkVulnerabilities(Domain $domain, SecurityCheck $securityCheck): void
    {
        $vulnerabilities = [];
        $riskLevel = 'low';

        try {
            // Check for common vulnerabilities
            $response = Http::timeout(30)->get('https://' . $domain->domain_name);
            $headers = $response->headers();
            $body = $response->body();

            // Check server header disclosure
            if (isset($headers['server'])) {
                $vulnerabilities[] = [
                    'type' => 'information_disclosure',
                    'description' => 'Server information disclosed in headers',
                    'details' => ['server' => $headers['server'][0]],
                    'risk' => 'low'
                ];
            }

            // Check for outdated technologies
            $this->checkOutdatedTechnologies($body, $vulnerabilities);

            // Determine overall risk level
            foreach ($vulnerabilities as $vuln) {
                if ($vuln['risk'] === 'high') {
                    $riskLevel = 'high';
                    break;
                } elseif ($vuln['risk'] === 'medium' && $riskLevel !== 'high') {
                    $riskLevel = 'medium';
                }
            }

            $securityCheck->update([
                'vulnerability_status' => $riskLevel === 'high' ? 'failed' : 'passed',
                'vulnerability_data' => [
                    'vulnerabilities' => $vulnerabilities,
                    'risk_level' => $riskLevel,
                    'count' => count($vulnerabilities)
                ]
            ]);

            if ($riskLevel === 'high') {
                $this->createSecurityAlert($domain, 'high_risk_vulnerabilities', 'High-risk vulnerabilities detected', [
                    'vulnerabilities' => array_filter($vulnerabilities, fn($v) => $v['risk'] === 'high'),
                    'risk_level' => $riskLevel
                ]);
            }

        } catch (\Exception $e) {
            $securityCheck->update([
                'vulnerability_status' => 'failed',
                'vulnerability_data' => ['error' => $e->getMessage()]
            ]);
        }
    }

    /**
     * Calculate overall security score.
     */
    protected function calculateSecurityScore(SecurityCheck $securityCheck): void
    {
        $weights = [
            'ssl' => 0.25,
            'malware' => 0.30,
            'blacklist' => 0.25,
            'headers' => 0.15,
            'vulnerability' => 0.05
        ];

        $scores = [
            'ssl' => $this->getStatusScore($securityCheck->ssl_status),
            'malware' => $this->getStatusScore($securityCheck->malware_status),
            'blacklist' => $this->getStatusScore($securityCheck->blacklist_status),
            'headers' => $this->getStatusScore($securityCheck->headers_status),
            'vulnerability' => $this->getStatusScore($securityCheck->vulnerability_status)
        ];

        $weightedScore = 0;
        foreach ($weights as $category => $weight) {
            $weightedScore += $scores[$category] * $weight;
        }

        $securityCheck->update([
            'security_score' => round($weightedScore, 2),
            'score_breakdown' => $scores
        ]);
    }

    /**
     * Get numeric score for status.
     */
    protected function getStatusScore(string $status): int
    {
        return match($status) {
            'passed' => 100,
            'warning' => 75,
            'failed' => 0,
            'pending' => 0,
            default => 0
        };
    }

    /**
     * Check if SSL certificate is self-signed.
     */
    protected function isSelfSigned(array $certData): bool
    {
        $issuer = $certData['issuer']['CN'] ?? '';
        $subject = $certData['subject']['CN'] ?? '';
        return $issuer === $subject;
    }

    /**
     * Check Google Safe Browsing.
     */
    protected function checkGoogleSafeBrowsing(Domain $domain): array
    {
        $apiKey = config('security.malware_scanning.google_safe_browsing.api_key');
        if (!$apiKey) {
            return ['threat_detected' => false, 'error' => 'API key not configured'];
        }

        try {
            $response = Http::timeout(30)->post(
                "https://safebrowsing.googleapis.com/v4/threatMatches:find?key={$apiKey}",
                [
                    'client' => [
                        'clientId' => 'trust-seal-portal',
                        'clientVersion' => '1.0'
                    ],
                    'threatInfo' => [
                        'threatTypes' => ['MALWARE', 'SOCIAL_ENGINEERING', 'UNWANTED_SOFTWARE'],
                        'platformTypes' => ['ANY_PLATFORM'],
                        'threatEntryTypes' => ['URL'],
                        'threatEntries' => [
                            ['url' => 'https://' . $domain->domain_name],
                            ['url' => 'http://' . $domain->domain_name]
                        ]
                    ]
                ]
            );

            $data = $response->json();
            $threatDetected = !empty($data['matches']);

            return [
                'threat_detected' => $threatDetected,
                'threats' => $data['matches'] ?? [],
                'service' => 'Google Safe Browsing'
            ];

        } catch (\Exception $e) {
            return ['threat_detected' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Check VirusTotal.
     */
    protected function checkVirusTotal(Domain $domain): array
    {
        $apiKey = config('security.malware_scanning.virustotal.api_key');
        if (!$apiKey) {
            return ['threat_detected' => false, 'error' => 'API key not configured'];
        }

        try {
            $url = 'https://' . $domain->domain_name;
            $urlId = base64_encode($url);
            $urlId = str_replace(['+', '/', '='], ['-', '_', ''], $urlId);

            $response = Http::timeout(30)
                ->withHeaders(['x-apikey' => $apiKey])
                ->get("https://www.virustotal.com/api/v3/urls/{$urlId}");

            if ($response->status() === 404) {
                // URL not found, submit for analysis
                $submitResponse = Http::timeout(30)
                    ->withHeaders(['x-apikey' => $apiKey])
                    ->asForm()
                    ->post('https://www.virustotal.com/api/v3/urls', ['url' => $url]);
                
                return ['threat_detected' => false, 'status' => 'submitted_for_analysis'];
            }

            $data = $response->json();
            $stats = $data['data']['attributes']['stats'] ?? [];
            $malicious = $stats['malicious'] ?? 0;
            $suspicious = $stats['suspicious'] ?? 0;

            return [
                'threat_detected' => $malicious > 0 || $suspicious > 2,
                'malicious_count' => $malicious,
                'suspicious_count' => $suspicious,
                'total_scans' => $stats['harmless'] + $stats['malicious'] + $stats['suspicious'] + $stats['undetected'],
                'service' => 'VirusTotal'
            ];

        } catch (\Exception $e) {
            return ['threat_detected' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Check URLVoid.
     */
    protected function checkUrlVoid(Domain $domain): array
    {
        $apiKey = config('security.malware_scanning.urlvoid.api_key');
        if (!$apiKey) {
            return ['threat_detected' => false, 'error' => 'API key not configured'];
        }

        try {
            $response = Http::timeout(30)->get('https://api.urlvoid.com/v1/pay-as-you-go/', [
                'key' => $apiKey,
                'host' => $domain->domain_name
            ]);

            $data = $response->json();
            $detections = $data['data']['report']['blacklists']['detections'] ?? 0;
            $engines = $data['data']['report']['blacklists']['engines_count'] ?? 0;

            return [
                'threat_detected' => $detections > 0,
                'detections' => $detections,
                'engines_count' => $engines,
                'detection_ratio' => $engines > 0 ? round(($detections / $engines) * 100, 2) : 0,
                'service' => 'URLVoid'
            ];

        } catch (\Exception $e) {
            return ['threat_detected' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Check blacklist service.
     */
    protected function checkBlacklistService(Domain $domain, string $service): array
    {
        return match($service) {
            'google_safe_browsing' => $this->checkGoogleSafeBrowsing($domain),
            'malware_domain_list' => $this->checkMalwareDomainList($domain),
            'phishtank' => $this->checkPhishTank($domain),
            'openphish' => $this->checkOpenPhish($domain),
            default => ['blacklisted' => false, 'error' => 'Unknown service']
        };
    }

    /**
     * Check for outdated technologies.
     */
    protected function checkOutdatedTechnologies(string $body, array &$vulnerabilities): void
    {
        // Check for jQuery versions
        if (preg_match('/jquery[.-]([0-9]+\.[0-9]+\.[0-9]+)/i', $body, $matches)) {
            $version = $matches[1];
            if (version_compare($version, '3.5.0', '<')) {
                $vulnerabilities[] = [
                    'type' => 'outdated_library',
                    'description' => 'Outdated jQuery version detected',
                    'details' => ['version' => $version, 'library' => 'jQuery'],
                    'risk' => 'medium'
                ];
            }
        }

        // Check for WordPress version disclosure
        if (preg_match('/wp-content\/themes|wp-includes/i', $body)) {
            if (preg_match('/wordpress[\s]*([0-9]+\.[0-9]+)/i', $body, $matches)) {
                $vulnerabilities[] = [
                    'type' => 'cms_detection',
                    'description' => 'WordPress version disclosed',
                    'details' => ['cms' => 'WordPress', 'version' => $matches[1] ?? 'unknown'],
                    'risk' => 'low'
                ];
            }
        }
    }

    /**
     * Create security alert.
     */
    protected function createSecurityAlert(Domain $domain, string $type, string $message, array $data = []): void
    {
        $alert = SecurityAlert::create([
            'domain_id' => $domain->id,
            'type' => $type,
            'level' => $this->getAlertLevel($type),
            'message' => $message,
            'data' => $data,
            'status' => 'unresolved',
            'detected_at' => now()
        ]);

        // Fire security alert event
        event(new SecurityAlertEvent($alert));
    }

    /**
     * Get alert level based on type.
     */
    protected function getAlertLevel(string $type): string
    {
        $highRiskTypes = ['malware_detected', 'domain_blacklisted', 'high_risk_vulnerabilities'];
        $mediumRiskTypes = ['ssl_expiring', 'ssl_self_signed', 'missing_security_headers'];
        
        if (in_array($type, $highRiskTypes)) {
            return 'critical';
        } elseif (in_array($type, $mediumRiskTypes)) {
            return 'warning';
        }
        
        return 'info';
    }

    /**
     * Get recent security checks for a domain.
     */
    public function getRecentChecks(Domain $domain, int $limit = 10): Collection
    {
        return SecurityCheck::where('domain_id', $domain->id)
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();
    }

    /**
     * Get security summary for a domain.
     */
    public function getSecuritySummary(Domain $domain): array
    {
        $latestCheck = SecurityCheck::where('domain_id', $domain->id)
            ->orderBy('created_at', 'desc')
            ->first();

        if (!$latestCheck) {
            return [
                'status' => 'unknown',
                'score' => 0,
                'last_check' => null,
                'alerts_count' => 0
            ];
        }

        $alertsCount = SecurityAlert::where('domain_id', $domain->id)
            ->where('status', 'unresolved')
            ->count();

        return [
            'status' => $latestCheck->status,
            'score' => $latestCheck->security_score ?? 0,
            'last_check' => $latestCheck->completed_at,
            'alerts_count' => $alertsCount,
            'ssl_status' => $latestCheck->ssl_status,
            'malware_status' => $latestCheck->malware_status,
            'blacklist_status' => $latestCheck->blacklist_status
        ];
    }

    /**
     * Check additional blacklist services.
     */
    protected function checkMalwareDomainList(Domain $domain): array
    {
        // This would typically check against malware domain lists
        return ['blacklisted' => false, 'service' => 'Malware Domain List'];
    }

    protected function checkPhishTank(Domain $domain): array
    {
        // This would check against PhishTank database
        return ['blacklisted' => false, 'service' => 'PhishTank'];
    }

    protected function checkOpenPhish(Domain $domain): array
    {
        // This would check against OpenPhish feeds
        return ['blacklisted' => false, 'service' => 'OpenPhish'];
    }
}