<?php

return [

    /*
    |--------------------------------------------------------------------------
    | Notification Channels Configuration
    |--------------------------------------------------------------------------
    |
    | This file contains configuration for various notification channels
    | available in the trust seal portal application.
    |
    */

    /*
    |--------------------------------------------------------------------------
    | Default Channels
    |--------------------------------------------------------------------------
    */

    'default' => [
        'mail',
        'database',
    ],

    /*
    |--------------------------------------------------------------------------
    | Mail Channel
    |--------------------------------------------------------------------------
    */

    'mail' => [
        'enabled' => env('MAIL_NOTIFICATIONS_ENABLED', true),
        'queue' => env('MAIL_QUEUE', 'emails'),
        'connection' => env('MAIL_QUEUE_CONNECTION', 'redis'),
        'from' => [
            'address' => env('MAIL_FROM_ADDRESS', 'noreply@trustsealportal.com'),
            'name' => env('MAIL_FROM_NAME', 'Trust Seal Portal'),
        ],
        'templates' => [
            'default' => 'emails.notification',
            'domain_verified' => 'emails.domain-verified',
            'security_alert' => 'emails.security-alert',
            'subscription' => 'emails.subscription',
            'ssl_expiry' => 'emails.ssl-expiry',
            'malware_detected' => 'emails.malware-detected',
            'welcome' => 'emails.welcome',
            'payment_success' => 'emails.payment-success',
            'payment_failed' => 'emails.payment-failed',
        ],
        'priorities' => [
            'urgent' => [
                'queue' => 'urgent-emails',
                'delay' => 0,
            ],
            'high' => [
                'queue' => 'high-emails',
                'delay' => 0,
            ],
            'normal' => [
                'queue' => 'emails',
                'delay' => 0,
            ],
            'low' => [
                'queue' => 'low-emails',
                'delay' => 300, // 5 minutes
            ],
        ],
        'retry' => [
            'attempts' => 3,
            'delay' => 60, // seconds
        ],
        'rate_limiting' => [
            'enabled' => true,
            'max_attempts' => 10,
            'decay_minutes' => 1,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Database Channel
    |--------------------------------------------------------------------------
    */

    'database' => [
        'enabled' => env('DATABASE_NOTIFICATIONS_ENABLED', true),
        'table' => 'notifications',
        'queue' => env('DATABASE_NOTIFICATION_QUEUE', 'database'),
        'connection' => env('DATABASE_NOTIFICATION_QUEUE_CONNECTION', 'redis'),
        'cleanup' => [
            'enabled' => true,
            'older_than_days' => 90,
            'keep_important' => true,
            'schedule' => 'daily',
        ],
        'indexing' => [
            'enabled' => true,
            'indexes' => [
                'user_id',
                'type',
                'read_at',
                'created_at',
                'priority',
            ],
        ],
        'priorities' => [
            'urgent' => [
                'retention_days' => 365,
                'auto_read' => false,
            ],
            'high' => [
                'retention_days' => 180,
                'auto_read' => false,
            ],
            'normal' => [
                'retention_days' => 90,
                'auto_read' => false,
            ],
            'low' => [
                'retention_days' => 30,
                'auto_read' => true,
            ],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Slack Channel
    |--------------------------------------------------------------------------
    */

    'slack' => [
        'enabled' => env('SLACK_NOTIFICATIONS_ENABLED', false),
        'webhook_url' => env('SLACK_WEBHOOK_URL'),
        'channel' => env('SLACK_CHANNEL', '#notifications'),
        'username' => env('SLACK_USERNAME', 'Trust Seal Portal'),
        'emoji' => env('SLACK_EMOJI', ':shield:'),
        'queue' => env('SLACK_QUEUE', 'slack'),
        'connection' => env('SLACK_QUEUE_CONNECTION', 'redis'),
        'templates' => [
            'default' => 'slack.notification',
            'security_alert' => 'slack.security-alert',
            'system_alert' => 'slack.system-alert',
            'payment_failed' => 'slack.payment-failed',
        ],
        'channels' => [
            'security' => env('SLACK_SECURITY_CHANNEL', '#security'),
            'billing' => env('SLACK_BILLING_CHANNEL', '#billing'),
            'system' => env('SLACK_SYSTEM_CHANNEL', '#system'),
            'general' => env('SLACK_GENERAL_CHANNEL', '#general'),
        ],
        'retry' => [
            'attempts' => 2,
            'delay' => 30, // seconds
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Discord Channel
    |--------------------------------------------------------------------------
    */

    'discord' => [
        'enabled' => env('DISCORD_NOTIFICATIONS_ENABLED', false),
        'webhook_url' => env('DISCORD_WEBHOOK_URL'),
        'username' => env('DISCORD_USERNAME', 'Trust Seal Portal'),
        'avatar_url' => env('DISCORD_AVATAR_URL'),
        'queue' => env('DISCORD_QUEUE', 'discord'),
        'connection' => env('DISCORD_QUEUE_CONNECTION', 'redis'),
        'embeds' => [
            'enabled' => true,
            'color' => env('DISCORD_EMBED_COLOR', 0x3498db),
            'footer' => [
                'text' => 'Trust Seal Portal',
                'icon_url' => env('DISCORD_FOOTER_ICON'),
            ],
        ],
        'retry' => [
            'attempts' => 2,
            'delay' => 30, // seconds
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | SMS Channel (Twilio)
    |--------------------------------------------------------------------------
    */

    'sms' => [
        'enabled' => env('SMS_NOTIFICATIONS_ENABLED', false),
        'driver' => env('SMS_DRIVER', 'twilio'),
        'queue' => env('SMS_QUEUE', 'sms'),
        'connection' => env('SMS_QUEUE_CONNECTION', 'redis'),
        'from' => env('SMS_FROM_NUMBER'),
        'providers' => [
            'twilio' => [
                'sid' => env('TWILIO_SID'),
                'token' => env('TWILIO_TOKEN'),
                'from' => env('TWILIO_FROM'),
            ],
            'nexmo' => [
                'key' => env('NEXMO_KEY'),
                'secret' => env('NEXMO_SECRET'),
                'from' => env('NEXMO_FROM'),
            ],
        ],
        'templates' => [
            'security_alert' => 'URGENT: Security issue detected on your domain {domain}. Please check your dashboard immediately.',
            'domain_verified' => 'Great! Your domain {domain} has been successfully verified.',
            'ssl_expiry' => 'Warning: SSL certificate for {domain} expires in {days} days.',
            'payment_failed' => 'Payment failed for your Trust Seal subscription. Please update your payment method.',
        ],
        'retry' => [
            'attempts' => 3,
            'delay' => 60, // seconds
        ],
        'rate_limiting' => [
            'enabled' => true,
            'max_attempts' => 5,
            'decay_minutes' => 1,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Push Notifications (Firebase)
    |--------------------------------------------------------------------------
    */

    'push' => [
        'enabled' => env('PUSH_NOTIFICATIONS_ENABLED', false),
        'driver' => env('PUSH_DRIVER', 'fcm'),
        'queue' => env('PUSH_QUEUE', 'push'),
        'connection' => env('PUSH_QUEUE_CONNECTION', 'redis'),
        'providers' => [
            'fcm' => [
                'key' => env('FCM_SERVER_KEY'),
                'sender_id' => env('FCM_SENDER_ID'),
            ],
            'apn' => [
                'certificate' => env('APN_CERTIFICATE'),
                'passphrase' => env('APN_PASSPHRASE'),
                'sandbox' => env('APN_SANDBOX', true),
            ],
        ],
        'templates' => [
            'security_alert' => [
                'title' => 'Security Alert',
                'body' => 'Security issue detected on {domain}',
                'icon' => 'security',
                'sound' => 'alert',
                'priority' => 'high',
            ],
            'domain_verified' => [
                'title' => 'Domain Verified',
                'body' => 'Your domain {domain} has been verified',
                'icon' => 'verified',
                'sound' => 'success',
                'priority' => 'normal',
            ],
        ],
        'retry' => [
            'attempts' => 3,
            'delay' => 30, // seconds
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Webhook Channel
    |--------------------------------------------------------------------------
    */

    'webhook' => [
        'enabled' => env('WEBHOOK_NOTIFICATIONS_ENABLED', false),
        'queue' => env('WEBHOOK_NOTIFICATION_QUEUE', 'webhooks'),
        'connection' => env('WEBHOOK_NOTIFICATION_QUEUE_CONNECTION', 'redis'),
        'endpoints' => [
            'default' => env('WEBHOOK_NOTIFICATION_URL'),
            'security' => env('WEBHOOK_SECURITY_URL'),
            'billing' => env('WEBHOOK_BILLING_URL'),
        ],
        'headers' => [
            'Content-Type' => 'application/json',
            'User-Agent' => 'Trust-Seal-Portal-Webhook/1.0',
            'X-Webhook-Source' => 'trust-seal-portal',
        ],
        'authentication' => [
            'type' => env('WEBHOOK_AUTH_TYPE', 'bearer'), // bearer, basic, signature
            'token' => env('WEBHOOK_AUTH_TOKEN'),
            'username' => env('WEBHOOK_AUTH_USERNAME'),
            'password' => env('WEBHOOK_AUTH_PASSWORD'),
            'signature_key' => env('WEBHOOK_SIGNATURE_KEY'),
        ],
        'timeout' => env('WEBHOOK_TIMEOUT', 30),
        'retry' => [
            'attempts' => 3,
            'delay' => 60, // seconds
        ],
        'verify_ssl' => env('WEBHOOK_VERIFY_SSL', true),
    ],

    /*
    |--------------------------------------------------------------------------
    | Channel Routing
    |--------------------------------------------------------------------------
    */

    'routing' => [
        'domain_verified' => [
            'mail' => ['priority' => 'normal'],
            'database' => ['priority' => 'normal'],
        ],
        'security_alert' => [
            'mail' => ['priority' => 'urgent'],
            'database' => ['priority' => 'urgent'],
            'slack' => ['channel' => 'security'],
            'sms' => ['condition' => 'severity:critical'],
        ],
        'subscription_created' => [
            'mail' => ['priority' => 'high'],
            'database' => ['priority' => 'normal'],
        ],
        'subscription_cancelled' => [
            'mail' => ['priority' => 'high'],
            'database' => ['priority' => 'normal'],
            'slack' => ['channel' => 'billing'],
        ],
        'payment_failed' => [
            'mail' => ['priority' => 'urgent'],
            'database' => ['priority' => 'high'],
            'slack' => ['channel' => 'billing'],
            'sms' => ['condition' => 'always'],
        ],
        'ssl_expiry_warning' => [
            'mail' => ['priority' => 'high'],
            'database' => ['priority' => 'normal'],
        ],
        'malware_detected' => [
            'mail' => ['priority' => 'urgent'],
            'database' => ['priority' => 'urgent'],
            'slack' => ['channel' => 'security'],
            'sms' => ['condition' => 'always'],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | User Preferences
    |--------------------------------------------------------------------------
    */

    'user_preferences' => [
        'enabled' => true,
        'defaults' => [
            'mail' => true,
            'database' => true,
            'sms' => false,
            'push' => false,
        ],
        'overrides' => [
            'security_alert' => [
                'mail' => true, // Always send security alerts via email
                'database' => true, // Always save to database
            ],
            'payment_failed' => [
                'mail' => true, // Always send payment failures via email
            ],
        ],
        'premium_features' => [
            'sms' => ['plans' => ['premium', 'enterprise']],
            'push' => ['plans' => ['basic', 'premium', 'enterprise']],
            'webhook' => ['plans' => ['premium', 'enterprise']],
            'slack' => ['plans' => ['enterprise']],
            'discord' => ['plans' => ['enterprise']],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Rate Limiting
    |--------------------------------------------------------------------------
    */

    'rate_limiting' => [
        'enabled' => true,
        'global' => [
            'max_per_minute' => 100,
            'max_per_hour' => 1000,
            'max_per_day' => 10000,
        ],
        'per_user' => [
            'max_per_minute' => 10,
            'max_per_hour' => 100,
            'max_per_day' => 500,
        ],
        'per_channel' => [
            'mail' => [
                'max_per_minute' => 20,
                'max_per_hour' => 200,
            ],
            'sms' => [
                'max_per_minute' => 5,
                'max_per_hour' => 50,
            ],
            'slack' => [
                'max_per_minute' => 10,
                'max_per_hour' => 100,
            ],
            'webhook' => [
                'max_per_minute' => 15,
                'max_per_hour' => 150,
            ],
        ],
        'backoff' => [
            'enabled' => true,
            'base_delay' => 60, // seconds
            'max_delay' => 3600, // seconds
            'multiplier' => 2,
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Monitoring & Analytics
    |--------------------------------------------------------------------------
    */

    'monitoring' => [
        'enabled' => env('NOTIFICATION_MONITORING_ENABLED', true),
        'metrics' => [
            'track_delivery' => true,
            'track_opens' => true,
            'track_clicks' => true,
            'track_bounces' => true,
            'track_complaints' => true,
        ],
        'storage' => [
            'driver' => 'database',
            'table' => 'notification_metrics',
            'retention_days' => 90,
        ],
        'alerts' => [
            'enabled' => true,
            'thresholds' => [
                'failure_rate' => 10, // percent
                'bounce_rate' => 5, // percent
                'complaint_rate' => 1, // percent
            ],
            'channels' => ['slack', 'mail'],
        ],
        'reporting' => [
            'enabled' => true,
            'schedule' => 'weekly',
            'recipients' => [
                env('ADMIN_EMAIL', 'admin@trustsealportal.com'),
            ],
            'include' => [
                'delivery_stats',
                'engagement_stats',
                'error_summary',
                'channel_performance',
            ],
        ],
    ],

    /*
    |--------------------------------------------------------------------------
    | Testing & Development
    |--------------------------------------------------------------------------
    */

    'testing' => [
        'enabled' => env('NOTIFICATION_TESTING_ENABLED', false),
        'log_only' => env('NOTIFICATION_LOG_ONLY', false),
        'test_recipients' => [
            'mail' => env('TEST_MAIL_RECIPIENT'),
            'sms' => env('TEST_SMS_RECIPIENT'),
        ],
        'mock_channels' => [
            'mail' => env('MOCK_MAIL_CHANNEL', false),
            'sms' => env('MOCK_SMS_CHANNEL', false),
            'slack' => env('MOCK_SLACK_CHANNEL', false),
            'push' => env('MOCK_PUSH_CHANNEL', false),
        ],
        'debug' => [
            'enabled' => env('NOTIFICATION_DEBUG', false),
            'log_payloads' => env('NOTIFICATION_LOG_PAYLOADS', false),
            'log_responses' => env('NOTIFICATION_LOG_RESPONSES', false),
        ],
    ],

];