<?php

use Illuminate\Support\Facades\Broadcast;
use App\Models\User;
use App\Models\Domain;
use App\Models\TrustSeal;

/*
|--------------------------------------------------------------------------
| Broadcast Channels
|--------------------------------------------------------------------------
|
| Here you may register all of the event broadcasting channels that your
| application supports. The given channel authorization callbacks are
| used to determine if an authenticated user can listen to the channel.
|
*/

// User-specific channels
Broadcast::channel('user.{userId}', function (User $user, $userId) {
    return (int) $user->id === (int) $userId;
});

// Domain-specific channels (only domain owner can listen)
Broadcast::channel('domain.{domainId}', function (User $user, $domainId) {
    return $user->domains()->where('id', $domainId)->exists();
});

// Trust seal channels (only seal owner can listen)
Broadcast::channel('seal.{sealId}', function (User $user, $sealId) {
    return $user->domains()
        ->whereHas('trustSeal', function ($query) use ($sealId) {
            $query->where('id', $sealId);
        })
        ->exists();
});

// Security alert channels (domain owner only)
Broadcast::channel('security.{domainId}', function (User $user, $domainId) {
    return $user->domains()->where('id', $domainId)->exists();
});

// Analytics channels (domain owner only)
Broadcast::channel('analytics.{domainId}', function (User $user, $domainId) {
    return $user->domains()->where('id', $domainId)->exists();
});

// Subscription channels (user only)
Broadcast::channel('subscription.{userId}', function (User $user, $userId) {
    return (int) $user->id === (int) $userId;
});

// Admin channels (admin users only)
Broadcast::channel('admin', function (User $user) {
    return $user->can('admin');
});

// System-wide notifications (all authenticated users)
Broadcast::channel('notifications', function (User $user) {
    return true; // All authenticated users can listen
});

// Premium user channels
Broadcast::channel('premium', function (User $user) {
    return $user->can('premium');
});

// Presence channels for real-time collaboration
Broadcast::channel('presence.dashboard.{userId}', function (User $user, $userId) {
    if ((int) $user->id === (int) $userId) {
        return [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'avatar' => $user->avatar_url,
            'plan' => $user->subscription_plan ?? 'free',
            'online_at' => now()->toISOString(),
        ];
    }
});

// Domain collaboration channels (for teams)
Broadcast::channel('presence.domain.{domainId}', function (User $user, $domainId) {
    if ($user->domains()->where('id', $domainId)->exists()) {
        return [
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'avatar' => $user->avatar_url,
            'role' => 'owner', // Could be extended for team roles
            'online_at' => now()->toISOString(),
        ];
    }
});

// System health monitoring (admin only)
Broadcast::channel('system.health', function (User $user) {
    return $user->can('admin') ? [
        'id' => $user->id,
        'name' => $user->name,
        'role' => 'admin',
        'online_at' => now()->toISOString(),
    ] : false;
});

// Real-time analytics updates
Broadcast::channel('analytics.realtime.{domainId}', function (User $user, $domainId) {
    return $user->domains()->where('id', $domainId)->exists();
});

// Security monitoring updates
Broadcast::channel('security.realtime.{domainId}', function (User $user, $domainId) {
    return $user->domains()->where('id', $domainId)->exists();
});

// Bulk operation progress tracking
Broadcast::channel('bulk.progress.{userId}', function (User $user, $userId) {
    return (int) $user->id === (int) $userId && $user->can('premium');
});

// Trust seal view tracking (for real-time stats)
Broadcast::channel('seal.views.{sealId}', function (User $user, $sealId) {
    return $user->domains()
        ->whereHas('trustSeal', function ($query) use ($sealId) {
            $query->where('id', $sealId);
        })
        ->exists();
});