<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\DomainController;
use App\Http\Controllers\TrustSealController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\SubscriptionController;
use App\Http\Controllers\AnalyticsController;
use App\Http\Controllers\SecurityController;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\WebhookController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| contains the "web" middleware group. Now create something great!
|
*/

// Public routes
Route::get('/', [HomeController::class, 'index'])->name('home');
Route::get('/features', [HomeController::class, 'features'])->name('features');
Route::get('/pricing', [HomeController::class, 'pricing'])->name('pricing');
Route::get('/about', [HomeController::class, 'about'])->name('about');
Route::get('/contact', [HomeController::class, 'contact'])->name('contact');
Route::post('/contact', [HomeController::class, 'contactSubmit'])->name('contact.submit')
    ->middleware('throttle:contact-form');
Route::get('/privacy', [HomeController::class, 'privacy'])->name('privacy');
Route::get('/terms', [HomeController::class, 'terms'])->name('terms');
Route::get('/help', [HomeController::class, 'help'])->name('help');
Route::get('/help/{article}', [HomeController::class, 'helpArticle'])->name('help.article');

// Trust seal display (public)
Route::get('/seal/{seal}', [TrustSealController::class, 'show'])
    ->name('seal.show')
    ->middleware('throttle:trust-seal');
Route::get('/seal/{seal}/embed', [TrustSealController::class, 'embed'])
    ->name('seal.embed')
    ->middleware('throttle:trust-seal');
Route::post('/seal/{seal}/track', [TrustSealController::class, 'track'])
    ->name('seal.track')
    ->middleware('throttle:trust-seal');

// Authentication routes
Auth::routes(['verify' => true]);

// Custom JSON-based authentication routes for demo
Route::get('/login', [HomeController::class, 'login'])->name('login');
Route::post('/login', [HomeController::class, 'loginPost'])->name('login.post');
Route::post('/logout', [HomeController::class, 'logout'])->name('logout');
Route::get('/dashboard', [HomeController::class, 'dashboard'])->name('dashboard');
Route::get('/admin', [HomeController::class, 'admin'])->name('admin');

// Demo domain management routes
Route::get('/domains/create', [HomeController::class, 'createDomain'])->name('domains.create');
Route::post('/domains', [HomeController::class, 'storeDomain'])->name('domains.store');

// Protected routes
Route::middleware(['auth', 'verified'])->group(function () {
    // Dashboard
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    
    // Domain management
    Route::resource('domains', DomainController::class);
    Route::post('/domains/{domain}/verify', [DomainController::class, 'verify'])
        ->name('domains.verify')
        ->middleware('throttle:domain-verification');
    Route::post('/domains/{domain}/regenerate-token', [DomainController::class, 'regenerateToken'])
        ->name('domains.regenerate-token');
    Route::patch('/domains/{domain}/toggle-status', [DomainController::class, 'toggleStatus'])
        ->name('domains.toggle-status');
    Route::get('/domains/{domain}/security', [DomainController::class, 'security'])
        ->name('domains.security');
    Route::post('/domains/{domain}/security-check', [DomainController::class, 'runSecurityCheck'])
        ->name('domains.security-check')
        ->middleware('throttle:security-checks');
    Route::get('/domains/{domain}/analytics', [DomainController::class, 'analytics'])->name('domains.analytics');
    
    // Trust seal management
    Route::get('/domains/{domain}/seal', [TrustSealController::class, 'edit'])->name('seal.edit');
    Route::patch('/domains/{domain}/seal', [TrustSealController::class, 'update'])->name('seal.update');
    Route::patch('/domains/{domain}/seal/toggle', [TrustSealController::class, 'toggle'])->name('seal.toggle');
    Route::get('/domains/{domain}/seal/stats', [TrustSealController::class, 'stats'])->name('seal.stats');
    Route::get('/domains/{domain}/seal/preview', [TrustSealController::class, 'preview'])->name('seal.preview');
    Route::get('/domains/{domain}/seal/code', [TrustSealController::class, 'getEmbedCode'])->name('seal.code');
    
    // Profile management
    Route::get('/profile', [ProfileController::class, 'show'])->name('profile.show');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');
    
    // Subscription management
    Route::get('/subscription', [SubscriptionController::class, 'show'])->name('subscription.show');
    Route::post('/subscription/subscribe', [SubscriptionController::class, 'subscribe'])->name('subscription.subscribe');
    Route::post('/subscription/cancel', [SubscriptionController::class, 'cancel'])->name('subscription.cancel');
    Route::get('/subscription/billing-portal', [SubscriptionController::class, 'billingPortal'])->name('subscription.billing-portal');
    
    // Analytics
    Route::get('/analytics', [AnalyticsController::class, 'index'])->name('analytics');
    Route::get('/analytics/domain/{domain}', [AnalyticsController::class, 'domain'])->name('analytics.domain');
    Route::get('/analytics/export', [AnalyticsController::class, 'export'])
        ->name('analytics.export')
        ->middleware('throttle:exports');
    Route::get('/analytics/reports', [AnalyticsController::class, 'reports'])->name('analytics.reports');
    
    // Security monitoring
    Route::prefix('security')->group(function () {
        Route::get('/', [SecurityController::class, 'index'])->name('security.index');
        Route::get('/alerts', [SecurityController::class, 'alerts'])->name('security.alerts');
        Route::get('/checks', [SecurityController::class, 'checks'])->name('security.checks');
        Route::post('/run-check', [SecurityController::class, 'runCheck'])
            ->name('security.run-check')
            ->middleware('throttle:security-checks');
        Route::get('/settings', [SecurityController::class, 'settings'])->name('security.settings');
        Route::patch('/settings', [SecurityController::class, 'updateSettings'])->name('security.update-settings');
    });
    
    // API Documentation (Premium users only)
    Route::middleware(['premium'])->group(function () {
        Route::get('/api-docs', function () {
            return view('api-docs');
        })->name('api-docs');
    });
});

// Premium features
Route::middleware(['premium'])->group(function () {
    // Advanced analytics
    Route::get('/analytics/advanced', [AnalyticsController::class, 'advanced'])->name('analytics.advanced');
    Route::get('/analytics/trends', [AnalyticsController::class, 'trends'])->name('analytics.trends');
    
    // Bulk operations
    Route::get('/bulk', [DashboardController::class, 'bulk'])->name('bulk.index');
    Route::post('/bulk/security-checks', [SecurityController::class, 'bulkSecurityChecks'])
        ->name('bulk.security-checks')
        ->middleware('throttle:bulk-operations');
    Route::post('/bulk/domain-updates', [DomainController::class, 'bulkUpdates'])
        ->name('bulk.domain-updates')
        ->middleware('throttle:bulk-operations');
    
    // API token management
    Route::prefix('api-tokens')->group(function () {
        Route::get('/', [DashboardController::class, 'apiTokens'])->name('api-tokens.index');
        Route::post('/', [DashboardController::class, 'createApiToken'])->name('api-tokens.store');
        Route::delete('/{token}', [DashboardController::class, 'deleteApiToken'])->name('api-tokens.destroy');
    });
});

// Webhook routes (for payment providers, etc.)
Route::post('/webhooks/stripe', [WebhookController::class, 'handleStripe'])
    ->name('webhooks.stripe')
    ->middleware('throttle:webhooks');
Route::post('/webhooks/security', [WebhookController::class, 'handleSecurity'])
    ->name('webhooks.security')
    ->middleware('throttle:webhooks');
Route::post('/webhooks/analytics', [WebhookController::class, 'handleAnalytics'])
    ->name('webhooks.analytics')
    ->middleware('throttle:webhooks');

// Admin routes
Route::middleware(['auth', 'verified', 'can:admin'])->prefix('admin')->group(function () {
    Route::get('/dashboard', [AdminController::class, 'dashboard'])->name('admin.dashboard');
    
    // User management
    Route::prefix('users')->group(function () {
        Route::get('/', [AdminController::class, 'users'])->name('admin.users.index');
        Route::get('/{user}', [AdminController::class, 'userDetails'])->name('admin.users.show');
        Route::patch('/{user}/status', [AdminController::class, 'updateUserStatus'])->name('admin.users.update-status');
        Route::delete('/{user}', [AdminController::class, 'deleteUser'])->name('admin.users.destroy');
    });
    
    // Domain management
    Route::prefix('domains')->group(function () {
        Route::get('/', [AdminController::class, 'domains'])->name('admin.domains.index');
        Route::get('/{domain}', [AdminController::class, 'domainDetails'])->name('admin.domains.show');
        Route::patch('/{domain}/force-verify', [AdminController::class, 'forceVerifyDomain'])->name('admin.domains.force-verify');
        Route::delete('/{domain}', [AdminController::class, 'forceDeleteDomain'])->name('admin.domains.force-delete');
    });
    
    // Security monitoring
    Route::prefix('security')->group(function () {
        Route::get('/', [AdminController::class, 'security'])->name('admin.security.index');
        Route::get('/alerts', [AdminController::class, 'securityAlerts'])->name('admin.security.alerts');
        Route::post('/run-global-check', [AdminController::class, 'runGlobalSecurityCheck'])
            ->name('admin.security.run-global-check');
    });
    
    // System management
    Route::prefix('system')->group(function () {
        Route::get('/', [AdminController::class, 'system'])->name('admin.system.index');
        Route::get('/health', [AdminController::class, 'systemHealth'])->name('admin.system.health');
        Route::post('/clear-cache', [AdminController::class, 'clearCache'])->name('admin.system.clear-cache');
        Route::post('/process-analytics', [AdminController::class, 'processAnalytics'])
            ->name('admin.system.process-analytics');
        Route::get('/logs', [AdminController::class, 'logs'])->name('admin.system.logs');
    });
    
    // Settings
    Route::prefix('settings')->group(function () {
        Route::get('/', [AdminController::class, 'settings'])->name('admin.settings.index');
        Route::patch('/', [AdminController::class, 'updateSettings'])->name('admin.settings.update');
    });
});

// Special routes for trust seal embedding (CORS enabled)
Route::middleware(['cors'])->prefix('embed')->group(function () {
    Route::get('/seal/{seal}', [TrustSealController::class, 'embedScript'])->name('embed.seal');
    Route::get('/seal/{seal}/css', [TrustSealController::class, 'embedCSS'])->name('embed.seal.css');
    Route::post('/seal/{seal}/track', [TrustSealController::class, 'embedTrack'])
        ->name('embed.seal.track')
        ->middleware('throttle:trust-seal');
});
